/**
 * \file pappsomspp/spectrum/spectrum.h
 * \date 15/3/2015
 * \author Olivier Langella
 * \brief basic spectrum
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#ifndef SPECTRUM_H_
#define SPECTRUM_H_ 1

#include <iostream>
#include <QDataStream>
#include <vector>
#include <memory>

#include "../mass_range.h"
using namespace std;

namespace pappso {

struct Peak {
    Peak()
        :   mz(0), intensity(0)
    {}
    //Peak(Peak && toCopy) : mz(toCopy.mz), intensity(toCopy.intensity) {};
    Peak (const Peak & toCopy);
    Peak (pappso_double mzin, pappso_double intensityin): mz(mzin), intensity(intensityin) {}
    Peak (pair<pappso_double, pappso_double> pairin): mz(pairin.first), intensity(pairin.second) {}



    pappso_double mz=0;
    pappso_double intensity=0;
};

QDataStream& operator<<(QDataStream& out, const Peak& peak);
QDataStream& operator>>(QDataStream& out, Peak& peak);

/**
 \class spectrum
 \brief Represents a MS spectrum (intensity/mz)
 */

class Spectrum;

QDataStream& operator<<(QDataStream& out, const Spectrum& spectrum);
QDataStream& operator>>(QDataStream& out, Spectrum& spectrum);

typedef std::shared_ptr<const Spectrum> SpectrumSp;

class Spectrum {

public:
    Spectrum();
    Spectrum(std::vector<pair<pappso_double, pappso_double>> & v_in);
    Spectrum(const Spectrum & toCopy);
    Spectrum(Spectrum && toCopy);//move constructor
    Spectrum& operator=(const Spectrum& other);
    Spectrum& operator=(Spectrum&&); //move assignment
    bool operator ==(const Spectrum &other) const; //comparison equal
    bool operator !=(const Spectrum &other) const; //comparison different
    virtual ~Spectrum();


    SpectrumSp makeSpectrumSp() const;

    typedef std::vector<Peak>::const_iterator const_iterator;

    unsigned int size() const {
        return _v_peaks.size();
    }
    const_iterator begin() const {
        return _v_peaks.begin();
    }
    const_iterator end() const {
        return _v_peaks.end();
    }

    /** @brief get first peak of spectrum
     */
    const Peak & front() const;

    /** @brief get last peak of spectrum
     */
    const Peak & back() const;

    /** @brief total ion current
     * The sum of all the separate ion currents carried by the ions of different m/z contributing to a complete mass spectrum or in a specified m/z range of a mass spectrum. MS:1000285
     * */
    const pappso_double getTotalIonCurrent() const;

    /** @brief get the maximum intensity peak
     *  aka base peak : peak of the signal of highest intensity in the mass spectrum.
     * */
    const Peak & getMaxIntensity() const;

    /** @brief get the lowest intensity peak
     * */
    const Peak & getLowestIntensity() const;

    const unsigned int getSpectrumSize() const;

    void debugPrintValues() const;

    void reserve(const unsigned int size);

    void clear();

    /** @brief filter spectrum to take N most intense peaks
     * @param n number of peaks to take
     * @result new spectrum containing the N most intense peaks
     * */
    Spectrum takeNmostIntense(unsigned int n) const;

    /** \brief filter spectrum to remove peaks with mz < mz_min
     * */
    Spectrum removeMzUnder(pappso_double mz_min)const;

    /** \brief filter spectrum to remove peaks less intense than percent threshold
     * */
    Spectrum removeIntensityPercentUnder(float percentIntMIn) const;

    /** \brief filter spectrum to remove peaks less intense than threshold
     * */
    Spectrum removeIntensityUnder(pappso_double minimum_intensity) const;

    Spectrum applyDynamicRange(pappso_double dynamic_range)const;
    Spectrum round() const;
    Spectrum floor() const;


    //Spectrum removeParent(const PeptideSp & peptideSp, unsigned int parent_ion_charge, pappso_double lower_dalton, pappso_double upper_dalton) const;
    /** \brief remove a mass range from the spectrum
     */
    Spectrum removeMassRange(const MassRange & remove_mass_range) const;
    Spectrum filterSum(const MassRange & mass_range) const;


    /** \brief remove isotopes as in X!Tandem algorithm
     * this method doesn't really remove isotopes: it cleans up multiple intensities within one Dalton
     * of each other.
     * */
    Spectrum xtandemDeisotope() const;

    /** \brief clean probable C13 isotopes as in X!Tandem algorithm
     * */
    Spectrum xtandemRemoveC13() const;

    void push_back(const Peak& mz_int_pair);

    /** @brief sort peaks by mz
     */
    void sortMz();

    /** @brief removes duplicated mz peaks (needs ordered peaks)
     * */
    void unique();

    bool equals(const Spectrum & spectrum_simple, PrecisionP precision) const;

    //friend QDataStream& operator<<(QDataStream& out, const Spectrum& spectrum);
    //friend QDataStream& operator>>(QDataStream& out, Spectrum& spectrum);

private:

    //pair of mz intensity
    std::vector<Peak> _v_peaks;

};


}
#endif /*SPECTRUM_H_*/

