/**
 * \file pappsomspp/peptide/peptidenaturalisotopelist.cpp
 * \date 8/3/2015
 * \author Olivier Langella
 * \brief peptide natural isotope model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/
//make test ARGS="-V -I 7,7"
#include<QDebug>
#include "peptidenaturalisotopelist.h"
#include "../pappsoexception.h"

namespace pappso {

PeptideNaturalIsotopeList::PeptideNaturalIsotopeList(const PeptideInterfaceSp & peptide, pappso_double minimu_ratio_to_compute): _peptide(peptide)
{
    qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList begin " << minimu_ratio_to_compute;
    int number_of_fixed_oxygen = _peptide.get()->getNumberOfIsotope(Isotope::O18)+ _peptide.get()->getNumberOfIsotope(Isotope::O17);
    int number_of_fixed_sulfur = _peptide.get()->getNumberOfIsotope(Isotope::S33)+ _peptide.get()->getNumberOfIsotope(Isotope::S34)+ _peptide.get()->getNumberOfIsotope(Isotope::S36);
    int number_of_fixed_nitrogen = _peptide.get()->getNumberOfIsotope(Isotope::N15);
    int number_of_fixed_hydrogen = _peptide.get()->getNumberOfIsotope(Isotope::H2);

    int total_carbon(_peptide.get()->getNumberOfAtom(AtomIsotopeSurvey::C)-_peptide.get()->getNumberOfIsotope(Isotope::C13));

// qDebug() << "total_carbon " << total_carbon;
// qDebug() << "total_sulfur " << total_sulfur;
    std::map<Isotope, int> map_isotope;
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::C13, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::H2, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::N15, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::O17, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::O18, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::S33, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::S34, 0));
    map_isotope.insert(
        std::pair<Isotope, int>(Isotope::S36, 0));

    for (int nbc13=0; nbc13 <= total_carbon; nbc13++) {
        map_isotope[Isotope::C13] = nbc13;
        PeptideNaturalIsotopeSp pepIsotope = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
        this->_peptide_natural_isotope_list.push_back(pepIsotope);
        if (pepIsotope.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
            break;
        }
    }
    std::list< PeptideNaturalIsotopeSp > temp_list;

    // ******************************************************************
    // Sulfur isotope list
    int total_sulfur(_peptide.get()->getNumberOfAtom(AtomIsotopeSurvey::S)-number_of_fixed_sulfur);
    std::list< PeptideNaturalIsotopeSp >::iterator it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        map_isotope = it->get()->getIsotopeMap();
        for (int nbS34=1; nbS34 <= total_sulfur; nbS34++) {
            map_isotope[Isotope::S34] = nbS34;
            PeptideNaturalIsotopeSp pepIsotope = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotope);
            if (pepIsotope.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //qDebug() << "peptide " << pepIsotope.get()->getFormula(1) << " " << pepIsotope.get()->getIntensityRatio(1);
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());

    //compute S33 abundance
    temp_list.resize(0);
    it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        //qDebug() << "peptide S33 " << it->get()->getFormula(1) << " " <<it->get()->getIntensityRatio(1);
        map_isotope = it->get()->getIsotopeMap();
        for (int nbS33=1; nbS33 <= (total_sulfur-map_isotope[Isotope::S34]); nbS33++) {
            map_isotope[Isotope::S33] = nbS33;
            PeptideNaturalIsotopeSp pepIsotopeS33 = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotopeS33);
            if (pepIsotopeS33.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());

    //compute S36 abundance
    temp_list.resize(0);
    it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        map_isotope = it->get()->getIsotopeMap();
        for (int nbS36=1; nbS36 <= (total_sulfur-map_isotope[Isotope::S34]-map_isotope[Isotope::S33]); nbS36++) {
            map_isotope[Isotope::S36] = nbS36;
            PeptideNaturalIsotopeSp pepIsotopeS36 = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotopeS36);
            if (pepIsotopeS36.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());

    // ******************************************************************




    // ******************************************************************
    // Hydrogen isotope list
    temp_list.resize(0);
    //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList total_hydrogen";
    int total_hydrogen(_peptide.get()->getNumberOfAtom(AtomIsotopeSurvey::H)-number_of_fixed_hydrogen);
    it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        // qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList getIsotopeMap " << it->getFormula(1) << " " << _peptide_natural_isotope_list.size();
        map_isotope = it->get()->getIsotopeMap();
        for (int nbH2=1; nbH2 <= total_hydrogen; nbH2++) {
            map_isotope[Isotope::H2] = nbH2;
            PeptideNaturalIsotopeSp pepIsotope = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotope);
            if (pepIsotope.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());
    // ******************************************************************









    // ******************************************************************
    // Oxygen isotope list
    temp_list.resize(0);
    //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList total_oxygen";
    unsigned int total_oxygen(_peptide.get()->getNumberOfAtom(AtomIsotopeSurvey::O)-number_of_fixed_oxygen);
    it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList getIsotopeMap " << it->getFormula(1) << " " << _peptide_natural_isotope_list.size();
        map_isotope = it->get()->getIsotopeMap();
        for (unsigned int nbO18=1; nbO18 <= total_oxygen; nbO18++) {
            //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList nbO18 " << nbO18;
            map_isotope[Isotope::O18] = nbO18;
            PeptideNaturalIsotopeSp pepIsotope = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotope);
            if (pepIsotope.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());
    // ******************************************************************



    // ******************************************************************
    // Nitrogen isotope list
    temp_list.resize(0);
    //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList total_nitrogen";
    unsigned int total_nitrogen(_peptide.get()->getNumberOfAtom(AtomIsotopeSurvey::N)-number_of_fixed_nitrogen);
    it = _peptide_natural_isotope_list.begin();
    while (it != _peptide_natural_isotope_list.end()) {
        //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList getIsotopeMap " << it->getFormula(1) << " " << _peptide_natural_isotope_list.size();
        map_isotope = it->get()->getIsotopeMap();
        for (unsigned int nbN15=1; nbN15 <= total_oxygen; nbN15++) {
            //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList nbN15 " << nbN15;
            map_isotope[Isotope::N15] = nbN15;
            PeptideNaturalIsotopeSp pepIsotope = std::make_shared<PeptideNaturalIsotope>(_peptide, map_isotope);
            temp_list.push_back(pepIsotope);
            if (pepIsotope.get()->getIntensityRatio(1) < minimu_ratio_to_compute) {
                //it++;
                break;
            }
        }
        it++;
    }
    _peptide_natural_isotope_list.insert(it, temp_list.begin(), temp_list.end());
    // ******************************************************************

    //qDebug() << "PeptideNaturalIsotopeList::PeptideNaturalIsotopeList end size="<<_peptide_natural_isotope_list.size();
}

PeptideNaturalIsotopeListSp PeptideNaturalIsotopeList::makePeptideNaturalIsotopeListSp() const {
    return std::make_shared<PeptideNaturalIsotopeList>(*this);
}

PeptideNaturalIsotopeList::PeptideNaturalIsotopeList(const PeptideNaturalIsotopeList& other): _peptide(other._peptide), _peptide_natural_isotope_list(other._peptide_natural_isotope_list)
{

}

PeptideNaturalIsotopeList::~PeptideNaturalIsotopeList()
{

}

const std::map<unsigned int, pappso_double> PeptideNaturalIsotopeList::getIntensityRatioPerIsotopeNumber() const {
    std::list< PeptideNaturalIsotopeSp >::const_iterator it = _peptide_natural_isotope_list.begin();
    std::map<unsigned int, pappso_double> map_isotope_number;

    while(it != _peptide_natural_isotope_list.end()) {
        unsigned int number = it->get()->getIsotopeNumber();
        std::pair<std::map<unsigned int,pappso_double>::iterator,bool> mapnew =  map_isotope_number.insert(std::pair<unsigned int,pappso_double>(number, 0));
        if (mapnew.second == false) {
            // mapit = map_isotope_number.insert(std::pair<unsigned int,pappso_double>(number, 0));
        }
        mapnew.first->second +=it->get()->getIntensityRatio(1);
        it++;
    }
    return map_isotope_number;
}





/** /brief get a sorted (by expected intensity) vector of isotopes of the same level
 *
 * */

std::vector<PeptideNaturalIsotopeSp> PeptideNaturalIsotopeList::getByIsotopeNumber(unsigned int isotope_number, unsigned int charge) const {
    std::vector<PeptideNaturalIsotopeSp> v_isotope_list;

    for(auto && isotopeSp :_peptide_natural_isotope_list) {
        if (isotopeSp.get()->getIsotopeNumber() == isotope_number) {
            v_isotope_list.push_back(isotopeSp);
        }
    }
    std::sort (v_isotope_list.begin(), v_isotope_list.end(), [charge](const PeptideNaturalIsotopeSp & m, const PeptideNaturalIsotopeSp & n) {

        return (m.get()->getIntensityRatio(charge) >n.get()->getIntensityRatio(charge));
    });
    return v_isotope_list;

}


/** /brief get a sorted (by expected intensity) vector of natural isotope average by isotope number
 *
 * */
std::vector<PeptideNaturalIsotopeAverageSp> getByIntensityRatioByIsotopeNumber(const PeptideInterfaceSp & peptide, unsigned int charge, PrecisionP precision,unsigned int isotopeNumber, pappso_double minimumIntensity) {

    qDebug() << "getByIntensityRatioByIsotopeNumber begin" ;
    unsigned int askedIsotopeRank;
    unsigned int maxAskedIsotopeRank=10;
    pappso_double cumulativeRatio=0;
    std::vector<PeptideNaturalIsotopeAverageSp> v_isotopeAverageList;
    std::vector<PeptideNaturalIsotopeAverageSp> v_isotopeAverageListResult;

    std::vector<unsigned int> previousIsotopeRank;
    bool isEmpty = false;
    for (askedIsotopeRank=1; (askedIsotopeRank < maxAskedIsotopeRank) && (!isEmpty); askedIsotopeRank++) {
        PeptideNaturalIsotopeAverage isotopeAverage(peptide, askedIsotopeRank, isotopeNumber, charge, precision);
        isEmpty = isotopeAverage.isEmpty();
        if (isEmpty) {
        }
        else {
            if (std::find(previousIsotopeRank.begin(), previousIsotopeRank.end(), isotopeAverage.getIsotopeRank()) == previousIsotopeRank.end())
            {   //not Found
                previousIsotopeRank.push_back(isotopeAverage.getIsotopeRank());
                v_isotopeAverageList.push_back(isotopeAverage.makePeptideNaturalIsotopeAverageSp());
            }
        }
    }
    if (v_isotopeAverageList.size()==0) return v_isotopeAverageListResult;

    qDebug() << "getByIntensityRatioByIsotopeNumber comp" ;
    std::sort (v_isotopeAverageList.begin(), v_isotopeAverageList.end(), [](const PeptideNaturalIsotopeAverageSp & m,const PeptideNaturalIsotopeAverageSp & n) {

        return (m.get()->getIntensityRatio() >n.get()->getIntensityRatio());
    });

    cumulativeRatio=0;
    auto it = v_isotopeAverageList.begin();
    v_isotopeAverageListResult.clear();
    qDebug() << "getByIntensityRatioByIsotopeNumber cumul" ;
    while ((it != v_isotopeAverageList.end()) && (cumulativeRatio < minimumIntensity)) {
        cumulativeRatio += it->get()->getIntensityRatio();
        v_isotopeAverageListResult.push_back(*it);
        it++;
    }

    qDebug() << "getByIntensityRatioByIsotopeNumber end" ;

    return v_isotopeAverageListResult;
}


/** /brief get a sorted (by expected intensity) vector of natural isotope average
 *
 * */
std::vector<PeptideNaturalIsotopeAverageSp> PeptideNaturalIsotopeList::getByIntensityRatio(unsigned int charge, PrecisionP precision, pappso_double minimumIntensity) const {

    qDebug() << "PeptideNaturalIsotopeList::getByIntensityRatio begin" ;

    std::map<unsigned int, pappso_double> map_isotope_number = PeptideNaturalIsotopeList::getIntensityRatioPerIsotopeNumber();
    std::vector<std::pair<unsigned int, pappso_double>> _pair_vector;
    for (std::pair<unsigned int, pappso_double> map_element :map_isotope_number) {
        _pair_vector.push_back(map_element);
    }
    std::sort (_pair_vector.begin(), _pair_vector.end(), [](const std::pair<unsigned int, pappso_double> & a,const std::pair<unsigned int, pappso_double> & b)-> bool {
        return (a.second > b.second);
    });

    unsigned int max = 0;
    unsigned int i=0;
    pappso::pappso_double cumulativeRatio=0;
    for (std::pair<unsigned int, pappso_double> & map_element :_pair_vector) {
        cumulativeRatio += map_element.second;
        if (cumulativeRatio < minimumIntensity) {
            max+=1;
        }
        else {
            break;
        }
    }
    max+=1;
    
    std::vector<PeptideNaturalIsotopeAverageSp> v_isotopeAverageListResult;


    for (std::pair<unsigned int, pappso_double> map_element :_pair_vector) {
        if (i >= max) break; //no more than 10 isotope numbers
        unsigned int  isotopeNumber = map_element.first;
        std::vector<PeptideNaturalIsotopeAverageSp> v_isotopeAverageListOneNumber = getByIntensityRatioByIsotopeNumber(this->_peptide, charge, precision,isotopeNumber, minimumIntensity);

        for (PeptideNaturalIsotopeAverageSp & isotope_average_sp :v_isotopeAverageListOneNumber) {
            v_isotopeAverageListResult.push_back(isotope_average_sp);
        }
        i++;
    }
    qDebug() << "PeptideNaturalIsotopeList::getByIntensityRatio end" ;

    std::sort (v_isotopeAverageListResult.begin(), v_isotopeAverageListResult.end(), [](const PeptideNaturalIsotopeAverageSp & m,const PeptideNaturalIsotopeAverageSp & n)-> bool {

        return (m.get()->getIntensityRatio() >n.get()->getIntensityRatio());
    });


    cumulativeRatio=0;
    auto it = v_isotopeAverageListResult.begin();

    std::vector<PeptideNaturalIsotopeAverageSp> v_isotopeAverageListResult_final;
    qDebug() << "getByIntensityRatioByIsotopeNumber cumul " << minimumIntensity;
    while ((it != v_isotopeAverageListResult.end()) && (cumulativeRatio < minimumIntensity)) {
        qDebug() << "getByIntensityRatioByIsotopeNumber cumul ratio " << cumulativeRatio;
        cumulativeRatio += it->get()->getIntensityRatio();
        v_isotopeAverageListResult_final.push_back(*it);
        it++;
    }

    return v_isotopeAverageListResult_final;
}

}

