
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "../pappsoexception.h"
#include "peptidenaturalisotopeaverage.h"

#include "peptidenaturalisotopelist.h"

namespace pappso {
PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage(const PeptideInterfaceSp & peptide, unsigned int askedIsotopeRank, unsigned int isotopeLevel, unsigned int charge, PrecisionP precision)
    :PeptideNaturalIsotopeAverage( PeptideNaturalIsotopeList(peptide), askedIsotopeRank, isotopeLevel,charge, precision)
{

}

PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage(const PeptideNaturalIsotopeList & isotopeList, unsigned int askedIsotopeRank, unsigned int isotope_number, unsigned int charge, PrecisionP precision)
    :_peptideSp(isotopeList.getPeptideInterfaceSp()),_isotopeLevel(isotope_number),_isotopeRank(askedIsotopeRank),_z(charge),_precision(precision)
{   // get the askedIsotopeRank :
    const std::vector<PeptideNaturalIsotopeSp> v_isotope_list(isotopeList.getByIsotopeNumber(isotope_number, _z));
    qDebug() << "PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage v_isotope_list.size()="<<v_isotope_list.size()<< " " << isotope_number<< " " << askedIsotopeRank;
    _abundanceRatio = 0;
    _averageMz = 0;
    if (askedIsotopeRank > v_isotope_list.size()) {
        //there is no isotope at this rank
        return;
        // throw PappsoException(QObject::tr("askedIsotopeRank greater than v_isotope_list.size() %1 vs %2").arg(askedIsotopeRank).arg(v_isotope_list.size()));
    }
    else if (askedIsotopeRank < 1) {
        throw PappsoException(QObject::tr("askedIsotopeRank must be 1 or more and not %1" ).arg(askedIsotopeRank));
    }
    PeptideNaturalIsotopeSp askedIsotope = v_isotope_list.at(askedIsotopeRank-1);

    //select neighbors in the precision range :
    MassRange mz_range(askedIsotope.get()->getMz(charge), precision);

    for (auto && isotopeSp : v_isotope_list) {
        //qDebug() << "PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage formula="<<isotopeSp.get()->getFormula(1)<< " ratio="<< isotopeSp.get()->getIntensityRatio(charge);
        if (mz_range.contains(isotopeSp.get()->getMz(charge))) {
            _peptide_natural_isotope_list.push_back(isotopeSp);
            _averageMz += (isotopeSp.get()->getMz(charge) * isotopeSp.get()->getIntensityRatio(charge));
            _abundanceRatio += isotopeSp.get()->getIntensityRatio(charge);
        }
    }
    if (_abundanceRatio == 0) {
        //there is no isotope
        return;
    }
    _averageMz = (_averageMz/_abundanceRatio);
    //sort neighbors to get the first isotope by intensity
    std::sort (_peptide_natural_isotope_list.begin(), _peptide_natural_isotope_list.end(), [charge](const PeptideNaturalIsotopeSp & m,const PeptideNaturalIsotopeSp & n) {
        return (m.get()->getIntensityRatio(charge) >n.get()->getIntensityRatio(charge));
    });

    _isotopeRank = 1;
    for (auto && isotopeSp : v_isotope_list) {
        if (isotopeSp.get()== _peptide_natural_isotope_list.at(0).get()) {
            break;
        }
        _isotopeRank++;
    }
    qDebug() << "PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage end";
}


PeptideNaturalIsotopeAverageSp PeptideNaturalIsotopeAverage::makePeptideNaturalIsotopeAverageSp() const {
    return std::make_shared<PeptideNaturalIsotopeAverage>(*this);
}

PeptideNaturalIsotopeAverage::PeptideNaturalIsotopeAverage(const PeptideNaturalIsotopeAverage& other): _peptideSp(other._peptideSp), _precision(other._precision) {

    _peptide_natural_isotope_list = other._peptide_natural_isotope_list;

    _averageMz = other._averageMz;
    _abundanceRatio= other._abundanceRatio;
    _isotopeLevel= other._isotopeLevel;
    _isotopeRank= other._isotopeRank;
    _z= other._z;
}

PeptideNaturalIsotopeAverage::~PeptideNaturalIsotopeAverage()
{

}
}
