/**
 * \file pappsomspp/peptide/peptidefragmention.cpp
 * \date 10/3/2015
 * \author Olivier Langella
 * \brief peptide ion model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include <QDebug>
#include "peptidefragmention.h"
#include "../pappsoexception.h"
#include "peptidenaturalisotopelist.h"
#include "peptiderawfragmentmasses.h"

namespace pappso {
PeptideFragmentIon::PeptideFragmentIon(const PeptideFragmentSp & sp_fragment,PeptideIon ion_type):PeptideFragmentIon(sp_fragment, ion_type, 0)
{
}

PeptideFragmentIon::PeptideFragmentIon(const PeptideFragmentSp & sp_fragment,PeptideIon ion_type, unsigned int number_of_neutral_phospho_loss):_sp_fragment(sp_fragment), _ion_type(ion_type)
{
    _mass = _sp_fragment.get()->getMass();
    PeptideDirection direction = _sp_fragment.get()->getPeptideIonDirection();
    if (direction !=  pappso::getPeptideIonDirection(_ion_type)) {
        throw PappsoException(QString("PeptideIon %1 is not an %2 fragment").arg(PeptideFragmentIon::getPeptideIonName
                              (_ion_type)).arg(PeptideFragment::getPeptideIonDirectionName(direction)));
    }
    _mass -= MASSH2O;
    switch (_ion_type) {
        // -MASSH2O
    case PeptideIon::yp :
        _number_of_neutral_phospho_loss = number_of_neutral_phospho_loss;
        _mass -= ((MASSH2O + MASSPHOSPHORYLATEDR) * _number_of_neutral_phospho_loss);
        ion_type = PeptideIon::y;
        break;
    case PeptideIon::bp :
        _number_of_neutral_phospho_loss = number_of_neutral_phospho_loss;
        _mass -= ((MASSH2O + MASSPHOSPHORYLATEDR) * _number_of_neutral_phospho_loss);
        ion_type = PeptideIon::b;
        break;
    default:
        break;
    }
    _mass += PeptideRawFragmentMasses::getDeltaMass(ion_type);

}

PeptideFragmentIon::PeptideFragmentIon(const PeptideFragmentIon& other):_sp_fragment(other._sp_fragment), _ion_type(other._ion_type)
{
    _mass = other._mass;
}


PeptideFragmentIon::PeptideFragmentIon(PeptideFragmentIon && toCopy) //move constructor
    :_sp_fragment(std::move(toCopy._sp_fragment)), _ion_type(toCopy._ion_type), _mass(toCopy._mass)
{
}

PeptideFragmentIon::~PeptideFragmentIon()
{

}


PeptideDirection PeptideFragmentIon::getPeptideIonDirection(PeptideIon ion_type) {
    return pappso::getPeptideIonDirection(ion_type);
}

const QString PeptideFragmentIon::getPeptideIonName() const {
    if ((_ion_type == PeptideIon::yp) || (_ion_type == PeptideIon::bp)) {
        return getPeptideIonName(_ion_type)+_number_of_neutral_phospho_loss;
    }
    return getPeptideIonName(_ion_type);
}

const QString PeptideFragmentIon::getPeptideIonName(PeptideIon _ion_type) {
    switch (_ion_type) {
    case PeptideIon::y :
        return "y";
        break;
    case PeptideIon::yp :
        return "yP";
        break;
    case PeptideIon::ystar :
        return "y*";
        break;
    case PeptideIon::yo :
        return "yO";
        break;
    case PeptideIon::bstar :
        return "b*";
        break;
    case PeptideIon::bo :
        return "bO";
        break;
    case PeptideIon::a :
        return "a";
        break;
    case PeptideIon::astar :
        return "a*";
        break;
    case PeptideIon::ao :
        return "aO";
        break;
    case PeptideIon::c :
        return "c";
        break;
        //SvgIon.moxygen - mN
    case PeptideIon::z :
        return "z";
        break;
    case PeptideIon::b :
        return "b";
        break;
    case PeptideIon::bp :
        return "bP";
        break;
    case PeptideIon::x :
        return "x";
        break;
    default:
        throw PappsoException(QString("PeptideIon name not implemented"));
        break;
    }
}

const QColor PeptideFragmentIon::getPeptideIonColor(PeptideIon _ion_type) {
    switch (_ion_type) {
    case PeptideIon::y :
        return QColor("red");
        break;
    case PeptideIon::yp :
        return QColor("red");
        break;
    case PeptideIon::ystar :
        return QColor("red");
        break;
    case PeptideIon::yo :
        return QColor("orange");
        break;
    case PeptideIon::x :
        return QColor("orange");
        break;
    case PeptideIon::bstar :
        return QColor("blue");
        break;
    case PeptideIon::bo :
        return QColor("#ff00ff");
        break;
    case PeptideIon::a :
        return QColor("green");
        break;
    case PeptideIon::astar :
        return QColor("green");
        break;
    case PeptideIon::ao :
        return QColor("green");
        break;
    case PeptideIon::c :
        return QColor("blue");
        break;
        //SvgIon.moxygen - mN
    case PeptideIon::z :
        return QColor("red");
        break;
    case PeptideIon::b :
        return QColor("blue");
        break;
    case PeptideIon::bp :
        return QColor("blue");
        break;
    default:
        throw PappsoException(QString("PeptideIon color not implemented %1").arg(getPeptideIonName(_ion_type)));
        break;
    }
}

int PeptideFragmentIon::getNumberOfAtom(AtomIsotopeSurvey atom) const {
    int number = _sp_fragment.get()->getNumberOfAtom(atom);
    int diff = 0;
    switch (atom) {
    case AtomIsotopeSurvey::C :
        switch (_ion_type) {
        case PeptideIon::y :
            break;
        case PeptideIon::yp :
            //H 1 O 3 P 1 + H 2 0
            break;
        case PeptideIon::ystar :
            // _mass -= MASSNH3;
            break;
        case PeptideIon::yo :
            //_mass -= MASSH2O;
            break;
        case PeptideIon::bstar :
            //_mass -= MASSH2O;
            //_mass -= MASSNH3;
            break;
        case PeptideIon::bo :
            // _mass -= MASSH2O;
            // _mass -= MASSH2O;
            break;
        case PeptideIon::a :
            //_mass -= MASSH2O;
            // _mass -= MASSCO;
            diff = -1;
            break;
        case PeptideIon::c :
            //_mass += MASSNH3;
            break;
            //SvgIon.moxygen - mN
        case PeptideIon::z :
            //_mass -= MASSH2O;
            // _mass += MASSOXYGEN - MASSNITROGEN - MPROTIUM;
            break;
        case PeptideIon::b :
            //_mass -= MASSH2O;
            break;
        case PeptideIon::bp :
            //H 1 O 3 P 1 + H 2 0
            break;

        case PeptideIon::astar :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSNH3;
            diff = -1;
            break;
        case PeptideIon::ao :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSH2O;
            diff = -1;
            break;
        case PeptideIon::x :
            // +MASSCO + MASSOXYGEN
            diff = +1;
            break;
        default:
            throw PappsoException(QString("PeptideIon name not implemented"));
            break;
        }
        break;
    case AtomIsotopeSurvey::H :
        switch (_ion_type) {
        case PeptideIon::y :
            break;
        case PeptideIon::yp :
            //H 1 O 3 P 1 + H 2 0
            diff = -3 * _number_of_neutral_phospho_loss;
            break;
        case PeptideIon::ystar :
            //_mass -= MASSNH3;
            diff = -3;
            break;
        case PeptideIon::yo :
            // _mass -= MASSH2O;
            diff = -2;
            break;
        case PeptideIon::bstar :
            //_mass -= MASSH2O;
            // _mass -= MASSNH3;
            diff = -5;
            break;
        case PeptideIon::bo :
            //_mass -= MASSH2O;
            //_mass -= MASSH2O;
            diff = -4;
            break;
        case PeptideIon::a :
            // _mass -= MASSH2O;
            // _mass -= MASSCO;
            diff = -2;
            break;
        case PeptideIon::c :
            //_mass += MASSNH3;
            diff = -3;
            break;
            //SvgIon.moxygen - mN
        case PeptideIon::z :
            //_mass -= MASSH2O;
            // _mass += MASSOXYGEN - MASSNITROGEN - MPROTIUM;
            diff = -3;
            break;
        case PeptideIon::b :
            //_mass -= MASSH2O;
            diff = -2;
            break;
        case PeptideIon::bp :
            //H 1 O 3 P 1 + H 2 0
            diff = -3 * _number_of_neutral_phospho_loss;
            break;


        case PeptideIon::astar :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSNH3;
            diff = -5;
            break;
        case PeptideIon::ao :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSH2O;
            diff = -4;
            break;
        case PeptideIon::x :
            // +MASSCO + MASSOXYGEN
            diff = -2;
            break;
        default:
            throw PappsoException(QString("PeptideIon name not implemented"));
            break;
        }
        break;
    case AtomIsotopeSurvey::N :
        switch (_ion_type) {
        case PeptideIon::y :
            break;
        case PeptideIon::yp :
            //H 1 O 3 P 1 + H 2 0
            break;

        case PeptideIon::ystar :
            //_mass -= MASSNH3;
            diff = -1;
            break;
        case PeptideIon::yo :
            //_mass -= MASSH2O;
            break;
        case PeptideIon::bstar :
            //_mass -= MASSH2O;
            //_mass -= MASSNH3;
            diff = -1;
            break;
        case PeptideIon::bo :
            //_mass -= MASSH2O;
            //_mass -= MASSH2O;
            break;
        case PeptideIon::a :
            // _mass -= MASSH2O;
            //_mass -= MASSCO;
            break;
        case PeptideIon::c :
            //_mass += MASSNH3;
            diff = -1;
            break;
            //SvgIon.moxygen - mN
        case PeptideIon::z :
            //_mass -= MASSH2O;
            //_mass += MASSOXYGEN - MASSNITROGEN - MPROTIUM;
            diff = -1;
            break;
        case PeptideIon::b :
            //_mass -= MASSH2O;
            break;
        case PeptideIon::bp :
            //H 1 O 3 P 1 + H 2 0
            break;


        case PeptideIon::astar :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSNH3;
            diff = -1;
            break;
        case PeptideIon::ao :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSH2O;
            break;
        case PeptideIon::x :
            // +MASSCO + MASSOXYGEN
            break;
        default:
            throw PappsoException(QString("PeptideIon name not implemented"));
            break;
        }
        break;
    case AtomIsotopeSurvey::O :
        switch (_ion_type) {
        case PeptideIon::y :
            break;
        case PeptideIon::yp :
            //H 1 O 3 P 1 + H 2 0
            diff = -4 * _number_of_neutral_phospho_loss;
            break;

        case PeptideIon::ystar :
            //_mass -= MASSNH3;
            break;
        case PeptideIon::yo :
            //_mass -= MASSH2O;
            diff = -1;
            break;
        case PeptideIon::bstar :
            // _mass -= MASSH2O;
            //_mass -= MASSNH3;
            diff = -1;
            break;
        case PeptideIon::bo :
            //_mass -= MASSH2O;
            //_mass -= MASSH2O;
            diff = -2;
            break;
        case PeptideIon::a :
            //_mass -= MASSH2O;
            //_mass -= MASSCO;
            diff = -2;
            break;
        case PeptideIon::c :
            //_mass += MASSNH3;
            break;
            //SvgIon.moxygen - mN
        case PeptideIon::z :
            //_mass -= MASSH2O;
            //_mass += MASSOXYGEN - MASSNITROGEN  - MPROTIUM;
            diff = -2;
            break;
        case PeptideIon::b :
            // _mass -= MASSH2O;
            diff = -1;
            break;
        case PeptideIon::bp :
            //H 1 O 3 P 1 + H 2 0
            diff = -4 * _number_of_neutral_phospho_loss;
            break;


        case PeptideIon::astar :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSNH3;
            diff = -2;
            break;
        case PeptideIon::ao :
            //_mass -= MASSH2O;
            // _mass = - MASSCO - MASSH2O;
            diff = -3;
            break;
        case PeptideIon::x :
            // +MASSCO + MASSOXYGEN
            diff = +1;
            break;
        default:
            throw PappsoException(QObject::tr("PeptideIon name not implemented"));
            break;
        }
        break;
    case AtomIsotopeSurvey::S :
        break;
    default :
        qDebug() << "PeptideFragmentIon::getNumberOfAtom(AtomIsotopeSurvey atom) NOT IMPLEMENTED";
    }
    return number+diff;
}


int PeptideFragmentIon::getNumberOfIsotope(Isotope isotope) const {
    int number = _sp_fragment.get()->getNumberOfIsotope(isotope);
    return number;

}

PeptideFragmentIonSp PeptideFragmentIon::makePeptideFragmentIonSp() const {
    return std::make_shared<PeptideFragmentIon>(*this);
}



}
