/**
 * \file pappsomspp/peptide/peptidefragment.h
 * \date 10/3/2015
 * \author Olivier Langella
 * \brief peptide fragment model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "peptidefragment.h"
#include "../pappsoexception.h"

namespace pappso {
PeptideFragment::PeptideFragment(const PeptideSp & sp_peptide, PeptideDirection direction, unsigned int size): _sp_peptide(sp_peptide), _direction(direction), _size(size)
{
    if (_direction == PeptideDirection::Nter) {
        std::vector<Aa>::const_iterator it(_sp_peptide.get()->begin());
        std::vector<Aa>::const_iterator end(_sp_peptide.get()->end());
        int i=0;
        while (i < _size) {
            _mass += it->getMass();
            it++;
            i++;
        }
        _nter_cter_cleavage = AaModification::getInstance("internal:Cter_hydrolytic_cleavage_HO");
    }
    else {
        std::vector<Aa>::const_reverse_iterator it(_sp_peptide.get()->rbegin());
        std::vector<Aa>::const_reverse_iterator end(_sp_peptide.get()->rend());
        int i=0;
        while (i < _size) {
            _mass += it->getMass();
            it++;
            i++;
        }
        _nter_cter_cleavage = AaModification::getInstance("internal:Nter_hydrolytic_cleavage_H");
    }
    _mass+=_nter_cter_cleavage->getMass();
}


PeptideFragment::PeptideFragment(const PeptideFragment& other): _sp_peptide(other._sp_peptide), _direction(other._direction), _size(other._size)
{
    _mass = other._mass;
}


PeptideFragment::PeptideFragment(PeptideFragment && toCopy) //move constructor
    :_sp_peptide(std::move(toCopy._sp_peptide)), _direction(toCopy._direction), _size(toCopy._size), _mass(toCopy._mass)
{
}


PeptideFragment::~PeptideFragment()
{

}

const QString PeptideFragment::getPeptideIonDirectionName(PeptideDirection direction) {
    switch (direction) {
    case PeptideDirection::Cter :
        return "Cter";
        break;
    case PeptideDirection::Nter :
        return "Nter";
        break;
    default:
        throw PappsoException(QString("direction name not implemented"));
        break;
    }
}

const QString PeptideFragment::getSequence() const {
    QString sequence = _sp_peptide.get()->getSequence();
    int diffSize = _sp_peptide.get()->size() - _size;
    if (_direction == PeptideDirection::Nter) {
        sequence = sequence.mid(0, _size);
    }
    else {
        sequence = sequence.mid(diffSize, _size);
    }
    return sequence;
}


int PeptideFragment::getNumberOfAtom(AtomIsotopeSurvey atom) const {
    int number = 0;
    if (_direction == PeptideDirection::Nter) {
        std::vector<Aa>::const_iterator it(_sp_peptide.get()->begin());
        std::vector<Aa>::const_iterator end(_sp_peptide.get()->end());
        int i=0;
        while (i < _size) {
            number += it->getNumberOfAtom(atom);
            it++;
            i++;
        }
    }
    else {
        std::vector<Aa>::const_reverse_iterator it(_sp_peptide.get()->rbegin());
        std::vector<Aa>::const_reverse_iterator end(_sp_peptide.get()->rend());
        int i=0;
        while (i < _size) {
            number += it->getNumberOfAtom(atom);
            it++;
            i++;
        }
    }
    number += _nter_cter_cleavage->getNumberOfAtom(atom);
    //qDebug() << "Aa::getMass() end " << mass;
    return number;

}

int PeptideFragment::getNumberOfIsotope(Isotope isotope) const {
    int number = 0;
    if (_direction == PeptideDirection::Nter) {
        std::vector<Aa>::const_iterator it(_sp_peptide.get()->begin());
        std::vector<Aa>::const_iterator end(_sp_peptide.get()->end());
        int i=0;
        while (i < _size) {
            number += it->getNumberOfIsotope(isotope);
            it++;
            i++;
        }
    }
    else {
        std::vector<Aa>::const_reverse_iterator it(_sp_peptide.get()->rbegin());
        std::vector<Aa>::const_reverse_iterator end(_sp_peptide.get()->rend());
        int i=0;
        while (i < _size) {
            number += it->getNumberOfIsotope(isotope);
            it++;
            i++;
        }
    }
    //number += _nter_cter_cleavage->getNumberOfIsotope(atom);
    //qDebug() << "Aa::getMass() end " << mass;
    return number;

}

}
