/**
 * \file pappsomspp/msrun/private/pwizmsrunreader.cpp
 * \date 29/05/2018
 * \author Olivier Langella
 * \brief MSrun file reader base on proteowizard library
 */

/*******************************************************************************
 * Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and
 *implementation
 ******************************************************************************/

#include "pwizmsrunreader.h"


#include <QDebug>
#include "../../pappsoexception.h"
#include "../../exception/exceptionnotfound.h"

namespace pappso
{

  QualifiedSpectrum getQualifiedMassSpectrumFromPwizSpectrumPtr(
    const SpectrumId &massSpectrumId, pwiz::msdata::CVID native_id_format,
    pwiz::msdata::Spectrum *p_spectrum, bool binary_data);
  pwiz::msdata::SpectrumPtr
  getPwizSpectrumPtr(pwiz::msdata::SpectrumList *p_spectrum_list,
                     std::size_t spectrum_index, bool binary_data);

  pwiz::msdata::SpectrumPtr
  getPwizSpectrumPtr(pwiz::msdata::SpectrumList *p_spectrum_list,
                     std::size_t spectrum_index, bool binary_data)
  {
    pwiz::msdata::SpectrumPtr simple_spectrum_pwiz;
    try
      {
        simple_spectrum_pwiz =
          p_spectrum_list->spectrum(spectrum_index, binary_data);
      }
    catch(std::runtime_error &error)
      {
        qDebug() << "getQualifiedMassSpectrumFromPwizMSDataFile error "
                 << error.what() << " " << typeid(error).name();
        throw ExceptionNotFound(
          QObject::tr("Pwiz spectrum index %1 not found in "
                      "MS file std::runtime_error :\n%2")
            .arg(spectrum_index)
            .arg(error.what()));
      }
    catch(std::exception &error)
      {
        qDebug() << "getQualifiedMassSpectrumFromPwizMSDataFile error "
                 << error.what() << " " << typeid(error).name();
        throw ExceptionNotFound(
          QObject::tr("Pwiz spectrum index %1 not found in MS file :\n%2")
            .arg(spectrum_index)
            .arg(error.what()));
      }
    if(simple_spectrum_pwiz.get() == nullptr)
      {
        throw ExceptionNotFound(
          QObject::tr(
            "Pwiz spectrum index %1 not found in MS file : null pointer")
            .arg(spectrum_index));
      }
    return simple_spectrum_pwiz;
  }


  QualifiedSpectrum
  getQualifiedMassSpectrumFromPwizSpectrumPtr(
    const SpectrumId &massSpectrumId, pwiz::msdata::CVID native_id_format,
    pwiz::msdata::Spectrum *p_spectrum, bool binary_data)
  {

    std::string env;
    env = setlocale(LC_ALL, "");
    setlocale(LC_ALL, "C");

    QualifiedSpectrum qspectrum(massSpectrumId);
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
             << " precursors.size() " << p_spectrum->precursors.size();
    if(p_spectrum->precursors.size() > 0)
      {
        pwiz::msdata::Precursor &precursor = *(p_spectrum->precursors.begin());

        qspectrum.setPrecursorScanNum(
          QString(pwiz::msdata::id::translateNativeIDToScanNumber(
                    native_id_format, precursor.spectrumID)
                    .c_str())
            .toULong());

        if(precursor.selectedIons.size() > 0)
          {
            pwiz::msdata::SelectedIon &ion = *(precursor.selectedIons.begin());

            // selected ion m/z
            pappso_double selected_ion_mz =
              QString(ion.cvParam(pwiz::cv::MS_selected_ion_m_z).value.c_str())
                .toDouble();
            // peak intensity
            pappso_double peak_intensity =
              QString(ion.cvParam(pwiz::cv::MS_peak_intensity).value.c_str())
                .toDouble();

            //  unsigned int test =
            //  QString(ion.cvParam(pwiz::cv::MS_1200_series_LC_MSD_SL).value.c_str()).toUInt();
            //  qDebug() << " tes "<< test;
            // charge state
            unsigned int charge_state =
              QString(ion.cvParam(pwiz::cv::MS_charge_state).value.c_str())
                .toUInt();


            qspectrum.setPrecursorMz(selected_ion_mz);
            qspectrum.setPrecursorIntensity(peak_intensity);
            qDebug() << " selected_ion_mz " << selected_ion_mz;
            qDebug() << " peak_intensity " << peak_intensity;
            qDebug() << " charge_state " << charge_state;
            if(charge_state > 0)
              {
                qspectrum.setPrecursorCharge(charge_state);
              }
          }
      }

    qDebug() << "getQualifiedMassSpectrumFromPwizMSDataFile msLevel";
    unsigned int msLevel(
      QString(p_spectrum->cvParam(pwiz::msdata::MS_ms_level).value.c_str())
        .toUInt());

    pappso_double retentionTime =
      QString(p_spectrum->scanList.scans[0]
                .cvParam(pwiz::msdata::MS_scan_start_time)
                .value.c_str())
        .toDouble();
    qspectrum.setRtInSeconds(retentionTime);
    qspectrum.setMsLevel(msLevel);

    /* for (pwiz::data::CVParam cv_param :ion.cvParams) {
         pwiz::msdata::CVID param_id = cv_param.cvid;
         qDebug() << param_id ;
         //qDebug() << cv_param.cvid.c_str();
         qDebug() << cv_param.name().c_str();
         qDebug() << cv_param.value.c_str();
     }*/

    if(binary_data)
      {
        // fill in MZIntensityPair vector for convenient access to binary data
        std::vector<pwiz::msdata::MZIntensityPair> pairs;
        p_spectrum->getMZIntensityPairs(pairs);


        Spectrum spectrum;

        // iterate through the m/z-intensity pairs
        for(std::vector<pwiz::msdata::MZIntensityPair>::const_iterator
              it  = pairs.begin(),
              end = pairs.end();
            it != end; ++it)
          {
            // qDebug() << "it->mz " << it->mz << " it->intensity" <<
            // it->intensity;
            spectrum.push_back(Peak(it->mz, it->intensity));
          }
        qDebug() << "getQualifiedMassSpectrumFromPwizMSDataFile spectrum size "
                 << spectrum.size();
        if(native_id_format == pwiz::cv::CVID::MS_Mascot_MGF_format)
          {
            // sort peaks by mz
            spectrum.sortMz();
          }
        // cout << "spectrum_simple size  " <<
        // spectrum_simple.getSpectrumSize()<< endl;

        // lc = localeconv ();
        // qDebug() << " env=" << localeconv () << " lc->decimal_point " <<
        // lc->decimal_point;
        SpectrumSp spectrum_sp = spectrum.makeSpectrumSp();
        qspectrum.setOriginalSpectrumSp(spectrum_sp);
      }
    setlocale(LC_ALL, env.c_str());
    return qspectrum;
  }


  QualifiedSpectrum
  PwizMsRunReader::getQualifiedSpectrumFromPwizMSDataFile(unsigned int scan_num)
  {

    std::string env;
    env              = setlocale(LC_ALL, "");
    struct lconv *lc = localeconv();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point "
             << lc->decimal_point;
    setlocale(LC_ALL, "C");
    pappso::SpectrumId massSpectrumId(_m_msrun_id, scan_num);
    /*
       vector< pwiz::msdata::SourceFilePtr >::iterator source_file_it  =
       _p_ms_data_file->fileDescription.sourceFilePtrs.begin();
       if (source_file_it !=
       _p_ms_data_file->fileDescription.sourceFilePtrs.end()) {
           qspectrum.setMassSpectrumId(MassSpectrumId(_m_msrun_id, scan_num));
       }
       else {
           qspectrum.setMassSpectrumId(MassSpectrumId(_m_msrun_id, scan_num));
       }
       */
    if(_m_p_msdatafile == nullptr)
      {
        return (QualifiedSpectrum(massSpectrumId));
      }
    const bool getBinaryData = true;
    pwiz::msdata::SpectrumListPtr spectrum_list_ptr =
      _m_p_msdatafile->run.spectrumListPtr;
    pwiz::msdata::CVID native_id_format;
    unsigned int spectrum_index = 0;
    if(spectrum_list_ptr.get()->size() == 1)
      {
        spectrum_index = 0;
      }
    else
      {
        // turn our scan numbers into navtieId strings for pwiz
        native_id_format =
          pwiz::msdata::id::getDefaultNativeIDFormat(*_m_p_msdatafile);
        std::string nativeScanId =
          pwiz::msdata::id::translateScanNumberToNativeID(
            native_id_format, QString("%1").arg(scan_num).toStdString());
        spectrum_index = spectrum_list_ptr.get()->find(nativeScanId);
      }
    qDebug() << "getQualifiedMassSpectrumFromPwizMSDataFile spectrum_index "
             << spectrum_index;
    if(spectrum_index == spectrum_list_ptr.get()->size())
      {
        // pwiz::msdata::SpectrumPtr simple_spectrum_pwiz =
        // spectrum_list_ptr.get()->spectrum(10, getBinaryData);
        // qDebug() << "spectrum id 10 " << simple_spectrum_pwiz.get()->get;
        throw ExceptionNotFound(
          QObject::tr("scan number %1 not found in MS file").arg(scan_num));
      }
    pwiz::msdata::SpectrumPtr simple_spectrum_pwiz = getPwizSpectrumPtr(
      spectrum_list_ptr.get(), spectrum_index, getBinaryData);

    setlocale(LC_ALL, env.c_str());

    return getQualifiedMassSpectrumFromPwizSpectrumPtr(
      massSpectrumId, native_id_format, simple_spectrum_pwiz.get(),
      getBinaryData);
  }


  PwizMsRunReader::PwizMsRunReader(const MsRunId &msrun_id)
    : pappso::MsRunReader(msrun_id)
  {
  }

  void
  PwizMsRunReader::initialize()
  {
    getPwizMSDataFile();
  }

  PwizMsRunReader::~PwizMsRunReader()
  {
    if(_m_p_msdatafile != nullptr)
      {
        delete _m_p_msdatafile;
      }
  }

  void
  PwizMsRunReader::getPwizMSDataFile()
  {
    QString filename(_m_msrun_id.getFilename());
    qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__ << " "
             << filename;
    std::string env;
    env              = setlocale(LC_ALL, "");
    struct lconv *lc = localeconv();
    qDebug() << " env=" << env.c_str() << " lc->decimal_point "
             << lc->decimal_point;
    setlocale(LC_ALL, "C");
    // lc = localeconv ();
    // qDebug() << " env=" << localeconv () << " lc->decimal_point " <<
    // lc->decimal_point;
    try
      {
        QByteArray byte_array = filename.toUtf8();
        std::string res       = "";
        for(char c : byte_array)
          {
            res += c;
          }
        _m_p_msdatafile = new pwiz::msdata::MSDataFile(res);
      }
    catch(std::exception &error)
      {
        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__;
        throw pappso::PappsoException(
          QObject::tr("Error reading file (%1) using proteowizard library : %2")
            .arg(filename)
            .arg(error.what()));
      }
    setlocale(LC_ALL, env.c_str());
  }


  pappso::SpectrumSp
  PwizMsRunReader::getSpectrumSp(unsigned int scan_num)
  {
    return getQualifiedSpectrum(scan_num).getOriginalSpectrumSp();
  }

  QualifiedSpectrum
  PwizMsRunReader::getQualifiedSpectrum(unsigned int scan_num)
  {

    QualifiedSpectrum spectrum;
    spectrum = getQualifiedSpectrumFromPwizMSDataFile(scan_num);
    if(_m_msrun_id.getMzFormat() == pappso::MzFormat::MGF)
      {
        if(spectrum.getRtInSeconds() == 0)
          {
            spectrum = getQualifiedSpectrumFromPwizMSDataFile(scan_num - 1);
          }
      }

    return spectrum;
  }


  void
  PwizMsRunReader::readSpectrumCollection(
    SpectrumCollectionHandlerInterface &handler)
  {


    const bool binary_data = handler.needPeakList();
    pwiz::msdata::SpectrumListPtr spectrum_list_ptr =
      _m_p_msdatafile->run.spectrumListPtr;
    pwiz::msdata::CVID native_id_format;
    std::size_t spectrum_list_size = spectrum_list_ptr.get()->size();
    native_id_format =
      pwiz::msdata::id::getDefaultNativeIDFormat(*_m_p_msdatafile);

    for(std::size_t i = 0; i < spectrum_list_size; i++)
      {
        pwiz::msdata::SpectrumPtr simple_spectrum_pwiz =
          getPwizSpectrumPtr(spectrum_list_ptr.get(), i, binary_data);
        /*
        if (simple_spectrum_pwiz->scanList.scans.size() == 0) {
            throw pappso::PappsoException(QObject::tr("scan number not found for
        spectrum index %1").arg(i));
        }
        if (simple_spectrum_pwiz->scanList.scans.size() > 1) {
            throw pappso::PappsoException(QObject::tr("unable to determine scan
        number for spectrum index %1 : too much scans").arg(i));
        }
        */

        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
                 << " spectrumID=" << simple_spectrum_pwiz->id.c_str();

        unsigned long scan_num =
          QString(pwiz::msdata::id::translateNativeIDToScanNumber(
                    native_id_format, simple_spectrum_pwiz->id)
                    .c_str())
            .toULong();

        qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __LINE__
                 << " scan_num=" << scan_num;
        SpectrumId massSpectrumId(_m_msrun_id, scan_num);
        handler.setQualifiedSpectrum(
          getQualifiedMassSpectrumFromPwizSpectrumPtr(
            massSpectrumId, native_id_format, simple_spectrum_pwiz.get(),
            binary_data));
      }
  }


  unsigned long
  PwizMsRunReader::getSpectrumListSize() const
  {
    return _m_p_msdatafile->run.spectrumListPtr.get()->size();
  }


} // namespace pappso
