/**
 * \file pappsomspp/msrun/msrunreader.h
 * \date 29/05/2018
 * \author Olivier Langella
 * \brief base interface to read MSrun files
 */

/*******************************************************************************
* Copyright (c) 2018 Olivier Langella <Olivier.Langella@u-psud.fr>.
*
* This file is part of the PAPPSOms++ library.
*
*     PAPPSOms++ is free software: you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation, either version 3 of the License, or
*     (at your option) any later version.
*
*     PAPPSOms++ is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
*
* Contributors:
*     Olivier Langella <Olivier.Langella@u-psud.fr> - initial API and implementation
******************************************************************************/

#ifndef MSRUNREADER_H
#define MSRUNREADER_H

#include <memory>
#include "msrunid.h"
#include "../spectrum/qualifiedspectrum.h"

namespace pappso {
    
    /** @brief interface to collect spectrums from the MsRunReader class
     */
class SpectrumCollectionHandlerInterface
{
public:
    virtual void setQualifiedSpectrum (const QualifiedSpectrum & qspectrum) = 0;
    virtual bool needPeakList () const =0;
};

   
/** @brief example of interface to count MS levels of all spectrum in an MSrun
 */
class MsRunSimpleStatistics: public SpectrumCollectionHandlerInterface
{
public:
    virtual void setQualifiedSpectrum (const QualifiedSpectrum & qspectrum) override;
    virtual bool needPeakList () const override ;
    
    unsigned long getMsLevelCount(unsigned int ms_level) const;
    
    unsigned long getTotalCount() const;
    
private:
    std::vector<unsigned long> _count_ms_level_spectrum;
};


class MsRunReader;
typedef std::shared_ptr<MsRunReader> MsRunReaderSp;

/** @brief base class to read MSrun
 * the only way to build a MsRunReader object is to use the MsRunReaderFactory
 */
class MsRunReader
{
public:
    /** @brief get a SpectrumSp class given its scan number
     */
    virtual SpectrumSp getSpectrumSp(unsigned int scan_num) = 0;
    
    /** @brief get a QualifiedSpectrum class given its scan number
     */
    virtual QualifiedSpectrum getQualifiedSpectrum(unsigned int scan_num) = 0;
    
    /** @brief get the totat number of spectrum conained in the MSrun data file
     */
    virtual unsigned long getSpectrumListSize() const = 0;
    
    /** @brief function to visit an MsRunReader and get each Spectrum in a spectrum collection handler
     */
    virtual void readSpectrumCollection(SpectrumCollectionHandlerInterface & handler) =0;
protected:
    MsRunReader(const MsRunId & msrun_id);
    MsRunReader(const MsRunReader & other);
    virtual ~MsRunReader();
    
    virtual void initialize() = 0;

protected:
    const MsRunId _m_msrun_id;
};
}
#endif // MSRUNREADER_H
