/**
 * \file pappsomspp/mass_range.h
 * \date 4/3/2015
 * \author Olivier Langella
 * \brief object to handle a mass range (an mz value + or - some delta)
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/
#ifndef _PAPPSOMSPP_MASS_RANGE_H_
#define _PAPPSOMSPP_MASS_RANGE_H_

#include "types.h"
#include <QString>
#include <map>

namespace pappso
{

class PrecisionBase
{
public:
    virtual PrecisionUnit getUnit() const = 0;
    virtual pappso_double getValue() const final;
    virtual pappso_double getDelta ( mz _mz ) const = 0;
    virtual QString toString() const = 0;

protected:
    PrecisionBase ( pappso_double x ) :
        _precision ( x )
    {
    }
protected :
    const pappso_double _precision;
};

/** \def specific type for a ppm precision
 *
 */
class PpmPrecision : public PrecisionBase
{
    friend class Precision;
public:
    virtual ~PpmPrecision() {};
    virtual PrecisionUnit getUnit() const;
    virtual pappso_double getDelta ( mz _mz ) const override
    {
        return ( ( _mz / ONEMILLION ) * _precision );
    };

    virtual QString toString() const override
    {
        return ( QString ( "%1 ppm" ).arg ( _precision ) );
    };

protected:
    PpmPrecision ( pappso_double x ) :
        PrecisionBase ( x )
    {
    }
};

/** \def specific type for a dalton precision
 *
 */
class DaltonPrecision :public PrecisionBase
{
    friend class Precision;
public:
    virtual ~DaltonPrecision() {};
    virtual PrecisionUnit getUnit() const;
    virtual pappso_double getDelta ( mz _mz ) const override
    {
        return _precision;
    };

    virtual QString toString() const override
    {
        return ( QString ( "%1 dalton" ).arg ( _precision ) );
    };

protected:
    DaltonPrecision ( pappso_double x ) :
        PrecisionBase ( x )
    {
    }

};


typedef const PrecisionBase * PrecisionP;



class Precision
{
public:
    static PrecisionP fromString ( const QString & precision_str );
    static PrecisionP getDaltonInstance ( pappso_double precision );
    static PrecisionP getPpmInstance ( pappso_double precision );

    using MapPpmPrecision =  std::map<pappso_double, PpmPrecision *>;
    using MapDaltonPrecision =  std::map<pappso_double, DaltonPrecision *>;

private :
    static MapPpmPrecision _map_ppm;
    static MapDaltonPrecision _map_dalton;
};


class MassRange
{

public:
    /// Constructs MassRange object using 1 precision (the same for lower or upper range).
    MassRange ( pappso::mz mz, PrecisionP precision );
    /// Constructs MassRange object using 2 different precisions: lower and upper.
    MassRange ( pappso::mz mz, PrecisionP precision_lower, PrecisionP precision_upper );

    MassRange ( const MassRange & other );

    const pappso::mz getMz() const;

    bool contains ( pappso::mz ) const;
    const QString toString() const;

    pappso::mz getLowest() const
    {
        return ( _mz - _delta );
    };
    pappso::mz getHighest() const
    {
        return ( _mz + _delta );
    };

private:
    pappso::mz _mz;
    pappso_double _delta;
};

}
#endif // _PAPPSOMSPP_MASS_RANGE_H_
