
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include <QDebug>
#include "grpsubgroup.h"
#include "grppeptideset.h"
#include "../pappsoexception.h"
#include "../utils.h"

using namespace pappso;

GrpSubGroup::GrpSubGroup(GrpProtein * p_protein) : _list_p_peptide(p_protein)
{
    _list_p_protein.push_back(p_protein);
}

GrpSubGroup::GrpSubGroup(const GrpSubGroup & other) : _list_p_peptide(other._list_p_peptide), _list_p_protein(other._list_p_protein)
{
}

unsigned int GrpSubGroup::getGroupNumber() const {
    return _group_number;
}
unsigned int GrpSubGroup::getSubGroupNumber() const {
    return _subgroup_number;
}
const std::vector<GrpProtein *> & GrpSubGroup::getGrpProteinList() const {
    return _list_p_protein;
}
const QString GrpSubGroup::getGroupingId() const {
    if(_group_number==0) {
        return "";
    }
    return QString("%1.%2").arg(Utils::getLexicalOrderedString(_group_number)).arg(Utils::getLexicalOrderedString(_subgroup_number));
}
unsigned int GrpSubGroup::maxCount() const {
    unsigned int max = 0;
    for (GrpProtein * p_protein:_list_p_protein) {
        if (max < p_protein->getCount()) {
            max = p_protein->getCount();
        }
    }
    return max;
}
bool GrpSubGroup::operator<(const GrpSubGroup& other) const {
    if (_list_p_peptide.size() == other._list_p_peptide.size()) {
        if (maxCount() == other.maxCount()) {
            if (_list_p_protein.size() == other._list_p_protein.size()) {
                //compare peptide set surface ?
                //alphabetic order taken into account
                return ((*(_list_p_protein.begin()))->getAccession() < (*(other._list_p_protein.begin()))->getAccession());
            }
            else {
                //if there is same peptide size evidence, then perhaps it's better to consider that
                // the best group is the one that include more proteins
                return (_list_p_protein.size() > other._list_p_protein.size());
            }
        }
        else {
            //counts are evidences of the presence of a subgroup
            //the fewer is the count, the weaker is the subgroup
            return (maxCount() > other.maxCount());
        }
    }
    else {
        //peptides are evidences of the presence of a subgroup
        //the fewer is the peptide list, the weaker is the subgroup
        return (_list_p_peptide.size() > other._list_p_peptide.size());
    }
}

GrpSubGroup::~GrpSubGroup()
{

}
GrpSubGroupSp GrpSubGroup::makeGrpSubGroupSp() {
    return std::make_shared<GrpSubGroup>(*this);

}

const GrpPeptideSet & GrpSubGroup::getPeptideSet() const {
    return _list_p_peptide;
}

bool GrpSubGroup::merge(GrpSubGroup * p_subgroup) {
    qDebug() << "GrpSubGroup::merge begin " << _list_p_protein.size() << " " << this->getFirstAccession() << " " << p_subgroup->getFirstAccession();
    //if (this == p_subgroup) {
    //     return true;
    //}
    if (p_subgroup->_list_p_peptide == _list_p_peptide) {
        //_list_p_protein.splice (_list_p_protein.end(), p_subgroup->_list_p_protein);
        _list_p_protein.insert(_list_p_protein.end(), p_subgroup->_list_p_protein.begin(), p_subgroup->_list_p_protein.end());
        //_list_p_protein.insert (_list_p_protein.end(), p_subgroup->_list_p_protein.begin(),p_subgroup->_list_p_protein.end());
        return true;
    }
    else {
        return false;
    }
}

bool GrpSubGroup::includes(const GrpSubGroup * p_subgroup) const {
    if (_list_p_peptide.biggerAndContainsAll(p_subgroup->getPeptideSet())) {
        return true;
    }
    else {
        return false;
    }
}

void GrpSubGroup::setGroupNumber(unsigned int i) {
    _group_number = i;
    for (auto && p_protein : _list_p_protein) {
        p_protein->setGroupNumber(i);
    }
}

void GrpSubGroup::setSubGroupNumber(unsigned int i) {
    _subgroup_number = i;
    for (auto && p_protein : _list_p_protein) {
        p_protein->setSubGroupNumber(i);
    }
}

void GrpSubGroup::numbering() {
    qDebug() << "GrpSubGroup::numbering begin";

    //sort proteins by accession numbers :
    //_list_p_protein.sort([](GrpProtein * first, GrpProtein * second) {
    //    return (first->getAccession() < second->getAccession()) ;
    //});
    std::sort(_list_p_protein.begin(), _list_p_protein.end(), [](GrpProtein * first, GrpProtein * second) {
        return (first->getAccession() < second->getAccession()) ;
    });
    //list unique removes all but the first element from every consecutive group of equal elements in the container
    //   _list_p_protein.unique();


    unsigned int i=1;
    for (auto && p_protein : _list_p_protein) {
        p_protein->setRank(i);
        i++;
    }
    qDebug() << "GrpSubGroup::numbering end";
}
const QString & GrpSubGroup::getFirstAccession() const {
    auto it =_list_p_protein.begin();
    if (it == _list_p_protein.end()) {
        throw PappsoException(QObject::tr("_list_p_protein is empty"));
    }
    else {
        return (*it)->getAccession();
    }
}

std::size_t GrpSubGroup::peptideListSize() const {
    return _list_p_peptide.size();
}


