
/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include <QDebug>
#include "grppeptideset.h"
#include "grpsubgroup.h"
#include "grppeptide.h"
#include "grpsubgroupset.h"
#include "grpgroup.h"
#include "grpexperiment.h"
#include "../pappsoexception.h"
#include "../utils.h"

using namespace pappso;

GrpGroup::GrpGroup(GrpSubGroupSp & grpSubGroupSp)
{
    _peptide_set.addAll(grpSubGroupSp.get()->getPeptideSet());
    _subgroup_list.push_back(grpSubGroupSp);
    _map_peptide2subgroup_set.add(grpSubGroupSp.get());
}
GrpGroup::GrpGroup(const GrpGroup & other): _peptide_set(other._peptide_set), _subgroup_list(other._subgroup_list), _map_peptide2subgroup_set(other._map_peptide2subgroup_set) {
}

GrpGroup::~GrpGroup()
{

}

std::vector<GrpSubGroupSpConst> GrpGroup::getGrpSubGroupSpList() const {
    std::vector<GrpSubGroupSpConst> subgroup_list;
    for (GrpSubGroupSp subgroup : _subgroup_list) {
        subgroup_list.push_back(subgroup);
    }
    return subgroup_list;
};

const std::list <GrpSubGroupSp> & GrpGroup::getSubGroupSpList() const {
    return _subgroup_list;
}

const GrpPeptideSet & GrpGroup::getGrpPeptideSet() const {
    return _peptide_set;
}
unsigned int GrpGroup::getGroupNumber() const {
    return _group_number;
}
const QString GrpGroup::getGroupingId() const {
    if(_group_number==0) {
        return "";
    }
    return QString("%1").arg(Utils::getLexicalOrderedString(_group_number));
}
bool GrpGroup::operator<(const GrpGroup& other) const {
    return ((*(_subgroup_list.begin()->get())) < (*(other._subgroup_list.begin()->get())));
}

GrpGroupSp GrpGroup::makeGrpGroupSp() {
    return std::make_shared<GrpGroup>(*this);
}

bool GrpGroup::containsAny(const GrpPeptideSet & peptideSet) const {

    return _peptide_set.containsAny(peptideSet);
}

void GrpGroup::addGroup(GrpGroup * p_group_to_add) {
    if (this == p_group_to_add) {
        throw PappsoException(QObject::tr("addGroup ERROR, this == p_group_to_add"));

    }

    for (GrpSubGroupSp & sgToAdd : p_group_to_add->_subgroup_list) {
        addSubGroupSp(sgToAdd);
    }
}

void GrpGroup::check() const {
    qDebug() << "GrpGroup::check begin " ;
    GrpSubGroupSet impacted_subgroup_list;
    _map_peptide2subgroup_set.getSubGroupSet(this->_peptide_set, impacted_subgroup_list);

    //if (impacted_subgroup_list.size() != this->_subgroup_list.size()) {
    qDebug() << "GrpGroup::check impacted_subgroup_list.size() != this->_subgroup_list.size()";
    qDebug() << impacted_subgroup_list.printInfos();

    for (auto sg : _subgroup_list) {
        qDebug() << sg->getFirstAccession()<< " " << sg.get();
    }
    //}
    qDebug() << _map_peptide2subgroup_set.printInfos();
    qDebug() << _peptide_set.printInfos();
    qDebug() << "GrpGroup::check end " ;
}


void GrpGroup::addSubGroupSp(const GrpSubGroupSp & grpSubGroupSp) {

    qDebug() << "GrpGroup::addSubGroupSp begin " << grpSubGroupSp.get()->getFirstAccession();


    //look for impacted subgroups (containing peptides alsoe present in incoming subgroup
    GrpSubGroupSet impacted_subgroup_list;
    _map_peptide2subgroup_set.getSubGroupSet(grpSubGroupSp.get()->getPeptideSet(), impacted_subgroup_list);
    if (impacted_subgroup_list.contains(grpSubGroupSp.get())) {
        throw PappsoException(QObject::tr("addSubGroupSp ERROR, subgroup %1 is already in group").arg(grpSubGroupSp.get()->getFirstAccession()));
    }

    //look for impacted subgroup equal to incoming subgroup and merge it and exit
    qDebug() << "GrpGroup::addSubGroupSp look for impacted subgroup equal to incoming subgroup and merge it and exit" ;
    auto it_impacted_sg = impacted_subgroup_list._list_p_subgroup.begin();
    auto it_impacted_sg_end = impacted_subgroup_list._list_p_subgroup.end();
    while (it_impacted_sg != it_impacted_sg_end) {

        if ((*it_impacted_sg)->merge(grpSubGroupSp.get())) {
            qDebug() << "GrpGroup::addSubGroupSp merge" ;
            return;
        }
        it_impacted_sg++;
    }

    //look for impacted subgroup including totally the incoming subgroup and exit
    qDebug() << "GrpGroup::addSubGroupSp look for impacted subgroup including totally the incoming subgroup and exit" ;
    it_impacted_sg = impacted_subgroup_list._list_p_subgroup.begin();
    while (it_impacted_sg != it_impacted_sg_end) {
        if ((*it_impacted_sg)->includes(grpSubGroupSp.get())) {
            qDebug() << "GrpGroup::addSubGroupSp includes" ;
            return;
        }
        it_impacted_sg++;
    }

    //look for impacted subgroup totally included in incoming subgroup and remove it
    qDebug() << "GrpGroup::addSubGroupSp look for impacted subgroup totally included in incoming subgroup and remove it" ;
    it_impacted_sg = impacted_subgroup_list._list_p_subgroup.begin();
    while (it_impacted_sg != it_impacted_sg_end) {
        if (grpSubGroupSp.get()->includes((*it_impacted_sg))) {
            // this->removeSubGroup(*it_impacted_sg);
            //it_impacted_sg = impacted_subgroup_list.erase(it_impacted_sg);
            GrpSubGroup * sg_to_remove = *it_impacted_sg;
            it_impacted_sg = impacted_subgroup_list.erase(it_impacted_sg);
            _map_peptide2subgroup_set.remove(sg_to_remove);
            _subgroup_list.remove_if([sg_to_remove](GrpSubGroupSp & sub_groupSp) {
                return (sg_to_remove == sub_groupSp.get()) ;
            });
            //_map_peptide2subgroup_set.check(_subgroup_list);
        }
        else {
            it_impacted_sg++;
        }
    }
    qDebug() << "GrpGroup::addSubGroupSp finally add the new subgroup to current group" ;

    //finally add the new subgroup to current group
    _subgroup_list.push_back(grpSubGroupSp);
    _map_peptide2subgroup_set.add(grpSubGroupSp.get());
    _peptide_set.addAll(grpSubGroupSp.get()->getPeptideSet());


    //check();
    // _map_peptide2subgroup_set.check(_subgroup_list);

    qDebug() << "GrpGroup::addSubGroupSp end" ;
}


void GrpGroup::setGroupNumber(unsigned int i) {
    qDebug() << "GrpGroup::setGroupNumber begin";
    _group_number = i;
    for (auto && sub_group_sp : _subgroup_list) {
        sub_group_sp->setGroupNumber(i);
    }
    _peptide_set.setGroupNumber(i);
    qDebug() << "GrpGroup::setGroupNumber end";
}


void GrpGroup::numbering() {
    qDebug() << "GrpGroup::numbering begin";

    for (auto && sub_group_sp : _subgroup_list) {
        sub_group_sp.get()->numbering();
    }
    _subgroup_list.sort([](GrpSubGroupSp & first, GrpSubGroupSp & second) {
        return ((* first.get()) < (* second.get())) ;
    });
    unsigned int i = 1;
    for (auto && sub_group_sp : _subgroup_list) {
        sub_group_sp.get()->setSubGroupNumber(i);
        i++;
    }

    _peptide_set.numbering();
    qDebug() << "GrpGroup::numbering end";
}


bool GrpGroup::removeFirstNonInformativeSubGroup() {
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup begin";
    std::list <GrpSubGroup *> subgroup_list_to_remove;

    for (auto && sub_group_sp : _subgroup_list) {
        if (_map_peptide2subgroup_set.hasSpecificPeptide(sub_group_sp.get())) {
        }
        else {
            subgroup_list_to_remove.push_back(sub_group_sp.get());
        }
    }
    if (subgroup_list_to_remove.size() == 0) {
        qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup end false";
        return false;
    }

    //sort subgroup to remove
    subgroup_list_to_remove.sort([](GrpSubGroup * first, GrpSubGroup * second) {
        return ((* first) < (* second)) ;
    });

    //remove the last one
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup subgroup_list_to_remove.front()->peptideListSize() " << subgroup_list_to_remove.front()->peptideListSize();
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup subgroup_list_to_remove.back()->peptideListSize() " << subgroup_list_to_remove.back()->peptideListSize();

    //the first subgroup is weaker (less peptides or less protein than others)
    GrpSubGroup * sg_to_remove = subgroup_list_to_remove.front();
    _map_peptide2subgroup_set.remove(sg_to_remove);
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup _subgroup_list.size() " << _subgroup_list.size();
    _subgroup_list.remove_if([sg_to_remove](GrpSubGroupSp & sub_groupSp) {
        return (sg_to_remove == sub_groupSp.get()) ;
    });
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup after remove if _subgroup_list.size() " << _subgroup_list.size();
    qDebug() << "GrpGroup::removeFirstNonInformativeSubGroup end true";
    return true;
}

bool GrpGroup::removeNonInformativeSubGroups() {
    qDebug() << "GrpGroup::removeNonInformativeSubGroups begin";
    if (removeFirstNonInformativeSubGroup()) {
        while (removeFirstNonInformativeSubGroup()) {
            qDebug() << "GrpGroup::removeNonInformativeSubGroups while";
        }
    }
    else {
        return false;
    }
    return true;
}

