/**
 * \file pappsomspp/amino_acid/aaBase.cpp
 * \date 7/3/2015
 * \author Olivier Langella
 * \brief private amino acid model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include<QDebug>
#include "aaBase.h"
#include "../exception/exceptionnotfound.h"

using namespace std;

namespace pappso {

aaBase::aaBase(char aa_letter) : _aa_letter(aa_letter) {
    /*
     if (aaBase::_aa_mass_map.empty()) {
     aaBase::static_builder();
     }
     */
    if (_aa_mass_map.find(_aa_letter) == _aa_mass_map.end()) {
        throw ExceptionNotFound(QObject::tr("amino acid %1 not found").arg(aa_letter));
    }
}


aaBase::aaBase(const aaBase & aa): _aa_letter(aa._aa_letter) {
}

aaBase::~aaBase() {
    // TODO Auto-generated destructor stub
}

aaBase::AaMassMap aaBase::_aa_mass_map = []
{
    AaMassMap ret;
    // populate ret
    //http://education.expasy.org/student_projects/isotopident/htdocs/aa-list.html

    // 	C11H12N2O2
    ret.insert(
        std::pair<char, pappso_double>('W', pappso_double(MASSCARBON * 11 + MPROTIUM * 10 + MASSNITROGEN *2 + MASSOXYGEN * 1)));
    //C2H5NO2
    ret.insert(
        std::pair<char, pappso_double>('G', pappso_double(MASSCARBON * 2 + MPROTIUM * 3 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    
    //C3H7NO2 
    ret.insert(
        std::pair<char, pappso_double>('A', pappso_double(MASSCARBON * 3 + MPROTIUM * 5 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C3H7NO3
    ret.insert(
        std::pair<char, pappso_double>('S', pappso_double(MASSCARBON * 3 + MPROTIUM * 5 + MASSNITROGEN *1 + MASSOXYGEN * 2)));
    //C5H9NO2
    ret.insert(
        std::pair<char, pappso_double>('P', pappso_double(MASSCARBON * 5 + MPROTIUM * 7 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C5H11NO2
    ret.insert(
        std::pair<char, pappso_double>('V', pappso_double(MASSCARBON * 5 + MPROTIUM * 9 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C4H9NO3
    ret.insert(
        std::pair<char, pappso_double>('T', pappso_double(MASSCARBON * 4 + MPROTIUM * 7 + MASSNITROGEN *1 + MASSOXYGEN * 2)));
    //C6H13NO2
    ret.insert(
        std::pair<char, pappso_double>('L', pappso_double(MASSCARBON * 6 + MPROTIUM * 11 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C6H13NO2
    ret.insert(
        std::pair<char, pappso_double>('I', pappso_double(MASSCARBON * 6 + MPROTIUM * 11 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C4H8N2O3
    ret.insert(
        std::pair<char, pappso_double>('N', pappso_double(MASSCARBON * 4 + MPROTIUM * 6 + MASSNITROGEN *2 + MASSOXYGEN * 2)));
    //C4H7NO4
    ret.insert(
        std::pair<char, pappso_double>('D', pappso_double(MASSCARBON * 4 + MPROTIUM * 5 + MASSNITROGEN *1 + MASSOXYGEN * 3)));
    //C6H14N2O2
    ret.insert(
        std::pair<char, pappso_double>('K', pappso_double(MASSCARBON * 6 + MPROTIUM * 12 + MASSNITROGEN *2 + MASSOXYGEN * 1)));
    //C5H10N2O3
    ret.insert(
        std::pair<char, pappso_double>('Q', pappso_double(MASSCARBON * 5 + MPROTIUM * 8 + MASSNITROGEN *2 + MASSOXYGEN * 2)));
    //C5H9NO4
    ret.insert(
        std::pair<char, pappso_double>('E', pappso_double(MASSCARBON * 5 + MPROTIUM * 7 + MASSNITROGEN *1 + MASSOXYGEN * 3)));
    
    //C5H11NO2S
    ret.insert(
        std::pair<char, pappso_double>('M', pappso_double(MASSCARBON * 5 + MPROTIUM * 9 + MASSNITROGEN *1 + MASSOXYGEN * 1 + MASSSULFUR)));
    // $arrret['m'] = 147.04; #METHIONINE OXIDEE (+16)
    //case 'm':
    //	mass = (float) 131.0404;
    //	addModification((float) 15.994915);
    //C6H9N3O2
    ret.insert(
        std::pair<char, pappso_double>('H', pappso_double(MASSCARBON * 6 + MPROTIUM * 7 + MASSNITROGEN *3 + MASSOXYGEN * 1)));
    //C9H11NO2
    ret.insert(
        std::pair<char, pappso_double>('F', pappso_double(MASSCARBON * 9 + MPROTIUM * 9 + MASSNITROGEN *1 + MASSOXYGEN * 1)));
    //C6H14N4O2
    ret.insert(
        std::pair<char, pappso_double>('R', pappso_double(MASSCARBON * 6 + MPROTIUM * 12 + MASSNITROGEN *4 + MASSOXYGEN * 1)));
    //C3H7NO2S
    ret.insert(
        std::pair<char, pappso_double>('C', pappso_double(MASSCARBON * 3 + MPROTIUM * 5 + MASSNITROGEN *1 + MASSOXYGEN * 1 + MASSSULFUR)));
    //mass = (float) 161.01; // CYSTEINE CARBAMIDOMETHYLE
    // addModification((float) 57.021464);
    //C9H11NO3
    ret.insert(
        std::pair<char, pappso_double>('Y', pappso_double(MASSCARBON * 9 + MPROTIUM * 9 + MASSNITROGEN *1 + MASSOXYGEN * 2)));
    
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, pappso_double>('U', pappso_double(168.964203)));
    //_aa_mass_map.insert(
    //		std::pair<char, pappso_double>('X', pappso_double(103.00919)));
    return ret;
}();

aaBase::AaIntMap aaBase::_aa_number_of_carbon_map = []
{
    AaIntMap ret;
    // populate ret

    ret.insert(
        std::pair<char, unsigned int>('W', 11));
    ret.insert(
        std::pair<char, unsigned int>('G', 2));
    ret.insert(
        std::pair<char, unsigned int>('A', 3));
    ret.insert(
        std::pair<char, unsigned int>('S', 3));
    ret.insert(
        std::pair<char, unsigned int>('P', 5));
    ret.insert(
        std::pair<char, unsigned int>('V', 5));
    ret.insert(
        std::pair<char, unsigned int>('T', 4));
    ret.insert(
        std::pair<char, unsigned int>('L', 6));
    ret.insert(
        std::pair<char, unsigned int>('I', 6));
    ret.insert(
        std::pair<char, unsigned int>('N', 4));
    ret.insert(
        std::pair<char, unsigned int>('D', 4));
    ret.insert(
        std::pair<char, unsigned int>('K', 6));
    ret.insert(
        std::pair<char, unsigned int>('Q', 5));
    ret.insert(
        std::pair<char, unsigned int>('E', 5));
    ret.insert(
        std::pair<char, unsigned int>('M', 5));
    ret.insert(
        std::pair<char, unsigned int>('H', 6));
    ret.insert(
        std::pair<char, unsigned int>('F', 9));
    ret.insert(
        std::pair<char, unsigned int>('R', 6));
    ret.insert(
        std::pair<char, unsigned int>('C', 3));
    ret.insert(
        std::pair<char, unsigned int>('Y', 9));
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, unsigned int>('U', 3));
    return ret;
}();


aaBase::AaIntMap aaBase::_aa_number_of_hydrogen_map = []
{
    AaIntMap ret;
    // populate ret

    ret.insert(
        std::pair<char, unsigned int>('A', 5));
    ret.insert(
        std::pair<char, unsigned int>('C', 5));
    ret.insert(
        std::pair<char, unsigned int>('D', 5));
    ret.insert(
        std::pair<char, unsigned int>('E', 7));
    ret.insert(
        std::pair<char, unsigned int>('F', 9));
    ret.insert(
        std::pair<char, unsigned int>('G', 3));
    ret.insert(
        std::pair<char, unsigned int>('H', 7));
    ret.insert(
        std::pair<char, unsigned int>('I', 11));
    ret.insert(
        std::pair<char, unsigned int>('K', 12));
    ret.insert(
        std::pair<char, unsigned int>('L', 11));
    ret.insert(
        std::pair<char, unsigned int>('M', 9));
    ret.insert(
        std::pair<char, unsigned int>('N', 6));
    ret.insert(
        std::pair<char, unsigned int>('P', 7));
    ret.insert(
        std::pair<char, unsigned int>('Q', 8));
    ret.insert(
        std::pair<char, unsigned int>('R', 12));
    ret.insert(
        std::pair<char, unsigned int>('S', 5));
    ret.insert(
        std::pair<char, unsigned int>('T', 7));
    ret.insert(
        std::pair<char, unsigned int>('V', 9));
    ret.insert(
        std::pair<char, unsigned int>('W', 10));
    ret.insert(
        std::pair<char, unsigned int>('Y', 9));
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, unsigned int>('U', 7));
    return ret;
}();


aaBase::AaIntMap aaBase::_aa_number_of_nitrogen_map = []
{
    AaIntMap ret;
    // populate ret

    ret.insert(
        std::pair<char, unsigned int>('A', 1));
    ret.insert(
        std::pair<char, unsigned int>('C', 1));
    ret.insert(
        std::pair<char, unsigned int>('D', 1));
    ret.insert(
        std::pair<char, unsigned int>('E', 1));
    ret.insert(
        std::pair<char, unsigned int>('F', 1));
    ret.insert(
        std::pair<char, unsigned int>('G', 1));
    ret.insert(
        std::pair<char, unsigned int>('H', 3));
    ret.insert(
        std::pair<char, unsigned int>('I', 1));
    ret.insert(
        std::pair<char, unsigned int>('K', 2));
    ret.insert(
        std::pair<char, unsigned int>('L', 1));
    ret.insert(
        std::pair<char, unsigned int>('M', 1));
    ret.insert(
        std::pair<char, unsigned int>('N', 2));
    ret.insert(
        std::pair<char, unsigned int>('P', 1));
    ret.insert(
        std::pair<char, unsigned int>('Q', 2));
    ret.insert(
        std::pair<char, unsigned int>('R', 4));
    ret.insert(
        std::pair<char, unsigned int>('S', 1));
    ret.insert(
        std::pair<char, unsigned int>('T', 1));
    ret.insert(
        std::pair<char, unsigned int>('V', 1));
    ret.insert(
        std::pair<char, unsigned int>('W', 2));
    ret.insert(
        std::pair<char, unsigned int>('Y', 1));
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, unsigned int>('U', 1));
    return ret;
}();

aaBase::AaIntMap aaBase::_aa_number_of_oxygen_map = []
{
    AaIntMap ret;
    // populate ret

    ret.insert(
        std::pair<char, unsigned int>('A', 1));
    ret.insert(
        std::pair<char, unsigned int>('C', 1));
    ret.insert(
        std::pair<char, unsigned int>('D', 3));
    ret.insert(
        std::pair<char, unsigned int>('E', 3));
    ret.insert(
        std::pair<char, unsigned int>('F', 1));
    ret.insert(
        std::pair<char, unsigned int>('G', 1));
    ret.insert(
        std::pair<char, unsigned int>('H', 1));
    ret.insert(
        std::pair<char, unsigned int>('I', 1));
    ret.insert(
        std::pair<char, unsigned int>('K', 1));
    ret.insert(
        std::pair<char, unsigned int>('L', 1));
    ret.insert(
        std::pair<char, unsigned int>('M', 1));
    ret.insert(
        std::pair<char, unsigned int>('N', 2));
    ret.insert(
        std::pair<char, unsigned int>('P', 1));
    ret.insert(
        std::pair<char, unsigned int>('Q', 2));
    ret.insert(
        std::pair<char, unsigned int>('R', 1));
    ret.insert(
        std::pair<char, unsigned int>('S', 2));
    ret.insert(
        std::pair<char, unsigned int>('T', 2));
    ret.insert(
        std::pair<char, unsigned int>('V', 1));
    ret.insert(
        std::pair<char, unsigned int>('W', 1));
    ret.insert(
        std::pair<char, unsigned int>('Y', 2));
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, unsigned int>('U', 2));
    return ret;
}();

aaBase::AaIntMap aaBase::_aa_number_of_sulfur_map = []
{
    AaIntMap ret;
    // populate ret

    ret.insert(
        std::pair<char, unsigned int>('A', 0));
    ret.insert(
        std::pair<char, unsigned int>('C', 1));
    ret.insert(
        std::pair<char, unsigned int>('D', 0));
    ret.insert(
        std::pair<char, unsigned int>('E', 0));
    ret.insert(
        std::pair<char, unsigned int>('F', 0));
    ret.insert(
        std::pair<char, unsigned int>('G', 0));
    ret.insert(
        std::pair<char, unsigned int>('H', 0));
    ret.insert(
        std::pair<char, unsigned int>('I', 0));
    ret.insert(
        std::pair<char, unsigned int>('K', 0));
    ret.insert(
        std::pair<char, unsigned int>('L', 0));
    ret.insert(
        std::pair<char, unsigned int>('M', 1));
    ret.insert(
        std::pair<char, unsigned int>('N', 0));
    ret.insert(
        std::pair<char, unsigned int>('P', 0));
    ret.insert(
        std::pair<char, unsigned int>('Q', 0));
    ret.insert(
        std::pair<char, unsigned int>('R', 0));
    ret.insert(
        std::pair<char, unsigned int>('S', 0));
    ret.insert(
        std::pair<char, unsigned int>('T', 0));
    ret.insert(
        std::pair<char, unsigned int>('V', 0));
    ret.insert(
        std::pair<char, unsigned int>('W', 0));
    ret.insert(
        std::pair<char, unsigned int>('Y', 0));
    // Selenocysteine	C3H7NO2Se 
    ret.insert(
        std::pair<char, unsigned int>('U', 0));
    return ret;
}();

pappso_double aaBase::getAaMass(char aa_letter) {
    return _aa_mass_map.at(aa_letter);
}


pappso_double aaBase::getMass() const {
    return _aa_mass_map.at(_aa_letter);
}


int aaBase::getNumberOfAtom(AtomIsotopeSurvey atom) const {
    switch (atom) {
    case AtomIsotopeSurvey::C :
        return this->_aa_number_of_carbon_map.at(_aa_letter);
    case AtomIsotopeSurvey::H :
        return this->_aa_number_of_hydrogen_map.at(_aa_letter);
    case AtomIsotopeSurvey::N :
        return this->_aa_number_of_nitrogen_map.at(_aa_letter);
    case AtomIsotopeSurvey::O :
        return this->_aa_number_of_oxygen_map.at(_aa_letter);
    case AtomIsotopeSurvey::S :
        return this->_aa_number_of_sulfur_map.at(_aa_letter);
    }
    //selenium (U) is not taken into account to compute isotopes
    // it has 5 stable isotopes and the most abundant is 80Se (49,61%)
    qDebug() << "aaBase::getNumberOfAtom(AtomIsotopeSurvey atom) NOT IMPLEMENTED";
    return 0;
}


void aaBase::replaceLeucineIsoleucine() {
  if (_aa_letter == 'L') _aa_letter = 'I';
}

} /* namespace pappso */
