/**
 * \file pappsomspp/amino_acid/aaBase.cpp
 * \date 7/3/2015
 * \author Olivier Langella
 * \brief amino acid model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "Aa.h"
#include <QDebug>
#include <QStringList>
#include <algorithm>

using namespace pappso;

Aa::Aa(char aa_letter): aaBase(aa_letter) {

}


Aa::Aa(const Aa & other): aaBase(other), _list_mod(other._list_mod) {
}


Aa::Aa(Aa && toCopy) //move constructor
    :  aaBase(toCopy), _list_mod(std::move(toCopy._list_mod))
{
}

Aa::~Aa() {
}

Aa& Aa::operator=(const Aa& toCopy) {
    _aa_letter = toCopy._aa_letter;
    _list_mod = toCopy._list_mod;
    return *this;
}

const std::list<AaModificationP> & Aa::getModificationList() const {
    return _list_mod;
}

pappso_double Aa::getMass() const {
    //qDebug() << "Aa::getMass() begin";
    pappso_double mass = aaBase::getMass();
    for (auto && mod : _list_mod) {
        mass += mod->getMass();
    }

    //qDebug() << "Aa::getMass() end " << mass;
    return mass;
}

const QString Aa::toAbsoluteString() const {
    QString seq = "";
    seq+=this->getLetter();
    auto it(_list_mod.begin());
    if (it != _list_mod.end()) {
        QStringList modification_str_list;
        while (it != _list_mod.end()) {
            modification_str_list << (*it)->getAccession();
            it++;
        }
        if (modification_str_list.size() > 0) seq+=QString("(%1)").arg(modification_str_list.join(","));
    }
    return seq;
}

const QString Aa::toString() const {
    QString seq = "";
    seq+=this->getLetter();
    auto it(_list_mod.begin());
    if (it != _list_mod.end()) {
        QStringList modification_str_list;
        while (it != _list_mod.end()) {
            if (!(*it)->isInternal()) {
                modification_str_list << (*it)->getAccession();
            }
            it++;
        }
        if (modification_str_list.size() > 0) seq+=QString("(%1)").arg(modification_str_list.join(","));
    }
    return seq;
}


void Aa::removeAaModification(AaModificationP mod) {
    _list_mod.remove(mod);
}

void  Aa::addAaModification(AaModificationP aaModification) {
    qDebug() << "Aa::addAaModification begin";
    qDebug() << aaModification->getAccession();
    _list_mod.push_back(aaModification);
    _list_mod.sort();
}

void Aa::replaceAaModification(AaModificationP oldmod, AaModificationP newmod) {
    unsigned int nbmod=getNumberOfModification(oldmod);
    _list_mod.remove(oldmod);
    for (unsigned int i=0; i < nbmod;i++) {
        _list_mod.push_back(newmod);
    }
    _list_mod.sort();
}

int Aa::getNumberOfAtom(AtomIsotopeSurvey atom) const {
    int number_of_carbon = aaBase::getNumberOfAtom(atom);
    for (auto && mod : _list_mod) {
        number_of_carbon += mod->getNumberOfAtom(atom);
    }

    //qDebug() << "Aa::getMass() end " << mass;
    return number_of_carbon;

}


int Aa::getNumberOfIsotope(Isotope isotope) const {
    int number = 0;
   for (auto && mod : _list_mod) {
        number += mod->getNumberOfIsotope(isotope);
    }

    //qDebug() << "Aa::getMass() end " << mass;
    return number;
}

unsigned int Aa::getNumberOfModification(AaModificationP mod) const {
    unsigned int number_of_mod = 0;
    for (auto && modb : _list_mod) {
        if(modb == mod) number_of_mod += 1;
    }

    //qDebug() << "Aa::getMass() end " << mass;
    return number_of_mod;

}

AaModificationP Aa::getInternalNterModification() const {
    for (auto && modb : _list_mod) {
        if(modb->getAccession().startsWith("internal:Nter_")) return modb;
    }
    return nullptr;
}
AaModificationP Aa::getInternalCterModification() const {
    for (auto && modb : _list_mod) {
        if(modb->getAccession().startsWith("internal:Cter_")) return modb;
    }
    return nullptr;
}
void Aa::removeInternalNterModification() {
    _list_mod.remove_if([](AaModificationP const & mod)
    {
        return mod->getAccession().startsWith("internal:Nter_");
    });
}
void Aa::removeInternalCterModification() {
    _list_mod.remove_if([](AaModificationP const & mod)
    {
        return mod->getAccession().startsWith("internal:Cter_");
    });
}

