/**
 * \file pappsomspp/spectrum/spectrum.h
 * \date 15/3/2015
 * \author Olivier Langella
 * \brief basic spectrum
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#ifndef SPECTRUM_H_
#define SPECTRUM_H_ 1

#include <iostream>
//#include <QDataStream>
#include <vector>
#include <memory>

#include "../mass_range.h"
using namespace std;

namespace pappso {


struct Peak {
    Peak()
        :   mz(0), intensity(0)
    {}
    //Peak(Peak && toCopy) : mz(toCopy.mz), intensity(toCopy.intensity) {};
    // Peak (const Peak & toCopy): mz(toCopy.mz), intensity(toCopy.intensity) {};
    Peak (pappso_double mzin, pappso_double intensityin): mz(mzin), intensity(intensityin) {}
    Peak (pair<pappso_double, pappso_double> pairin): mz(pairin.first), intensity(pairin.second) {}



    pappso_double mz=0;
    pappso_double intensity=0;
};

/**
 \class spectrum
 \brief Represents a MS spectrum (intensity/mz)
 */

class Spectrum;
typedef std::shared_ptr<const Spectrum> SpectrumSp;

class Spectrum {

public:
    Spectrum();
    Spectrum(std::vector<pair<pappso_double, pappso_double>> & v_in);
    Spectrum(const Spectrum & toCopy);
    Spectrum(Spectrum && toCopy);//move constructor
    Spectrum& operator=(const Spectrum& other);
    Spectrum& operator=(Spectrum&&); //move assignment
    virtual ~Spectrum();


    SpectrumSp makeSpectrumSp() const;

    typedef std::vector<Peak>::const_iterator const_iterator;

    unsigned int size() const {
        return _v_peaks.size();
    }
    const_iterator begin() const {
        return _v_peaks.begin();
    }
    const_iterator end() const {
        return _v_peaks.end();
    }
    const Peak front() const {
        return _v_peaks.front();
    }
    const Peak back() const {
        return _v_peaks.back();
    }

    const Peak getMaxIntensity() const;

    const unsigned int getSpectrumSize() const;

    void debugPrintValues() const;

    void reserve(const unsigned int size);

    void clear();

    /** \brief filter spectrum to take N most intense peaks
     * */
    Spectrum takeNmostIntense(unsigned int n) const;
    
    /** \brief filter spectrum to remove peaks with mz < mz_min
     * */
    Spectrum removeMzUnder(pappso_double mz_min)const;
    
    /** \brief filter spectrum to remove peaks less intense than percent threshold
     * */
    Spectrum removeIntensityPercentUnder(float percentIntMIn) const;
    
    /** \brief filter spectrum to remove peaks less intense than threshold
     * */
    Spectrum removeIntensityUnder(pappso_double minimum_intensity) const;
    
    Spectrum applyDynamicRange(pappso_double dynamic_range)const;
    Spectrum round() const;
    Spectrum floor() const;
    
    
    //Spectrum removeParent(const PeptideSp & peptideSp, unsigned int parent_ion_charge, pappso_double lower_dalton, pappso_double upper_dalton) const;
    /** \brief remove a mass range from the spectrum
     */
    Spectrum removeMassRange(const MassRange & remove_mass_range) const;
    Spectrum filterSum(const MassRange & mass_range) const;
    
    
    /** \brief remove isotopes as in X!Tandem algorithm
     * this method doesn't really remove isotopes: it cleans up multiple intensities within one Dalton
     * of each other.
     * */
    Spectrum xtandemDeisotope() const;
    
    /** \brief clean probable C13 isotopes as in X!Tandem algorithm
     * */
    Spectrum xtandemRemoveC13() const;

    void push_back(const Peak& mz_int_pair);

    bool equals(const Spectrum & spectrum_simple, PrecisionP precision) const;
private:

    //pair of mz intensity
    std::vector<Peak> _v_peaks;

};

}
#endif /*SPECTRUM_H_*/

