/*******************************************************************************
 * * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 * *
 * * This file is part of MassChroqPRM.
 * *
 * *     MassChroqPRM is free software: you can redistribute it and/or modify
 * *     it under the terms of the GNU General Public License as published by
 * *     the Free Software Foundation, either version 3 of the License, or
 * *     (at your option) any later version.
 * *
 * *     MassChroqPRM is distributed in the hope that it will be useful,
 * *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 * *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * *     GNU General Public License for more details.
 * *
 * *     You should have received a copy of the GNU General Public License
 * *     along with MassChroqPRM.  If not, see <http://www.gnu.org/licenses/>.
 * *
 * * Contributors:
 * *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 * ******************************************************************************/

#ifndef XIC_H
#define XIC_H


#include <vector>
#include <memory>
#include "../mass_range.h"

using namespace std;
namespace pappso {


struct XicElement {
    XicElement()
        :   rt(0), intensity(0)
    {}
    XicElement (pappso_double rtin, pappso_double intensityin): rt(rtin), intensity(intensityin) {}
    XicElement (std::pair<pappso_double, pappso_double> pairin): rt(pairin.first), intensity(pairin.second) {}

    XicElement (const XicElement & other): rt(other.rt), intensity(other.intensity) {}


    bool operator ==(const XicElement &b) const {
        return ((rt == b.rt) && (intensity == b.intensity));
    };
    bool operator <(const XicElement &b) const {
        return intensity < b.intensity;
    };

    pappso_double rt=0;
    pappso_double intensity=0;
};

class XicPeak;

class Xic;
typedef std::shared_ptr<const Xic> XicSp;
typedef std::shared_ptr<Xic> NoConstXicSp;


class Xic
{
public:
    Xic();
    virtual ~Xic();
    Xic(const Xic & toCopy);
    Xic(Xic && toCopy);
    Xic& operator=(const Xic& other);
    Xic& operator=(Xic&&); //move assignment

    bool operator !=(const Xic &b) const {
        return (!equal(b));
    };

    bool equal(const Xic &b) const;

    XicSp makeXicSp() const;
    
    NoConstXicSp makeNoConstXicSp() const;

    const XicElement& at (pappso_double rt) const;
    
    /** \brief get the number of MS measurement between 2 retention times on this xic */
    unsigned int getMsPointDistance(pappso_double rt, pappso_double rt_other) const;

    typedef std::vector<XicElement>::const_iterator const_iterator;

    unsigned int size() const {
        return _v_xic_elements.size();
    }

    std::vector<XicElement>::iterator begin() {
        return _v_xic_elements.begin();
    }
    std::vector<XicElement>::iterator end() {
        return _v_xic_elements.end();
    }
    const_iterator begin() const {
        return _v_xic_elements.begin();
    }
    const_iterator end() const {
        return _v_xic_elements.end();
    }

    const XicElement getMaxIntensity() const;

    void debugPrintValues() const;

    void reserve(const unsigned int size);

    void clear();
    void push_back(const XicElement& rt_int_pair);

//protected :
    void newXicByApplyingWindowsOperation( unsigned int half_edge_size,
                                           pappso_double(* p_sub_function)(const_iterator begin, const_iterator end), Xic & new_xic) const;
    void findPeakRangeWalkingFromMax(std::vector< XicElement >::const_iterator current_rt, XicPeak & peak) const;
    void integratePeakSurface(XicPeak & peak) const;

protected:
    //pair of mz intensity
    std::vector<XicElement> _v_xic_elements;

};
}
#endif // XIC_H
