/**
 * \file pappsomspp/psm/xtandem/xtandemhyperscore.cpp
 * \date 16/8/2016
 * \author Olivier Langella
 * \brief process spectrum to compute X!Tandem hyperscore
 */

/*******************************************************************************
 * Copyright (c) 2016 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "xtandemspectrumprocess.h"
#include <QDebug>

namespace pappso {
XtandemSpectrumProcess::XtandemSpectrumProcess()
{

}

XtandemSpectrumProcess::XtandemSpectrumProcess(const XtandemSpectrumProcess & copy) {
    _dynamic_range = copy._dynamic_range;
    _minimum_mz = copy._minimum_mz;
    _nmost_intense = copy._nmost_intense;
    _remove_isotope = copy._remove_isotope;
    _exclude_parent_neutral_loss = copy._exclude_parent_neutral_loss;
    _neutral_loss_mass = copy._neutral_loss_mass;
    _neutral_loss_window_dalton = copy._neutral_loss_window_dalton;

    _exclude_parent = copy._exclude_parent;
    _exclude_parent_lower_dalton = copy._exclude_parent_lower_dalton;
    _exclude_parent_upper_dalton = copy._exclude_parent_upper_dalton;
    _refine_spectrum_model = copy._refine_spectrum_model;
    _y_ions = copy._y_ions; //PeptideIon::y
    _b_ions = copy._b_ions; //PeptideIon::b
    _ystar_ions = copy._ystar_ions; //PeptideIon::ystar
    _bstar_ions = copy._bstar_ions; //PeptideIon::bstar
    _c_ions = copy._c_ions; //PeptideIon::c
    _z_ions = copy._z_ions; //PeptideIon::z
    _a_ions = copy._a_ions; //PeptideIon::a
    _x_ions = copy._x_ions; //CO2

    _astar_ions = copy._astar_ions; //PeptideIon::a
    _ao_ions = copy._ao_ions;
    _bo_ions =  copy._bo_ions;
    _yo_ions =  copy._yo_ions;

}
XtandemSpectrumProcess::~XtandemSpectrumProcess()
{

}

void XtandemSpectrumProcess::setMinimumMz(pappso::mz minimum_mz) {
    _minimum_mz = minimum_mz;
}
void XtandemSpectrumProcess::setNmostIntense(unsigned int nmost_intense) {
    _nmost_intense = nmost_intense;
}
void XtandemSpectrumProcess::setDynamicRange(pappso::pappso_double dynamic_range) {
    _dynamic_range = dynamic_range;
}

void XtandemSpectrumProcess::setRemoveIsotope(bool remove_isotope) {
    _remove_isotope = remove_isotope;
}

void XtandemSpectrumProcess::setExcludeParent(bool exclude_parent) {
    _exclude_parent = exclude_parent;
}
void XtandemSpectrumProcess::setExcludeParentNeutralLoss(bool neutral_loss) {
    _exclude_parent_neutral_loss = neutral_loss;
}
void XtandemSpectrumProcess::setNeutralLossMass(pappso::pappso_double neutral_loss_mass) {
    _neutral_loss_mass = neutral_loss_mass;
}
void XtandemSpectrumProcess::setNeutralLossWindowDalton(pappso_double neutral_loss_precision) {
    _neutral_loss_window_dalton = neutral_loss_precision;
}


void XtandemSpectrumProcess::setRefineSpectrumModel(bool refine) {
    _refine_spectrum_model = refine;
}
void XtandemSpectrumProcess::setIonScore(PeptideIon ion_type, bool compute_score) {
    switch (ion_type) {
    case PeptideIon::y:
        _y_ions = compute_score;
        break;

    case PeptideIon::b:
        _b_ions = compute_score;
        break;

    case PeptideIon::ystar:
        _ystar_ions = compute_score;
        break;

    case PeptideIon::bstar:
        _bstar_ions = compute_score;
        break;

    case PeptideIon::yo:
        _yo_ions = compute_score;
        break;

    case PeptideIon::bo:
        _bo_ions = compute_score;
        break;

    case PeptideIon::z:
        _z_ions = compute_score;
        break;

    case PeptideIon::a:
        _a_ions = compute_score;
        break;

    case PeptideIon::astar:
        _astar_ions = compute_score;
        break;
    case PeptideIon::ao:
        _ao_ions = compute_score;
        break;
    case PeptideIon::c:
        _c_ions = compute_score;
        break;
    case PeptideIon::x:
        _x_ions = compute_score;
        break;
    }
}

pappso::mz XtandemSpectrumProcess::getMinimumMz() const {
    return _minimum_mz ;
}
unsigned int XtandemSpectrumProcess::getNmostIntense() const {
    return _nmost_intense;
}
pappso::pappso_double XtandemSpectrumProcess::getDynamicRange() const {
    return _dynamic_range;
}
bool XtandemSpectrumProcess::getRemoveIsotope() const {
    return _remove_isotope;
}
bool XtandemSpectrumProcess::getExcludeParent() const {
    return _exclude_parent;
}
bool XtandemSpectrumProcess::getExcludeParentNeutralLoss() const {
    return _exclude_parent_neutral_loss;
}
pappso::pappso_double XtandemSpectrumProcess::getNeutralLossMass() const {
    return _neutral_loss_mass;
}
pappso_double XtandemSpectrumProcess::getNeutralLossWindowDalton() const {
    return _neutral_loss_window_dalton;

}

bool XtandemSpectrumProcess::getRefineSpectrumModel() const {
    return _refine_spectrum_model;
}
bool XtandemSpectrumProcess::getIonScore(PeptideIon ion_type) const {
    switch (ion_type) {
    case PeptideIon::y:
        return _y_ions;

    case PeptideIon::b:
        return _b_ions ;

    case PeptideIon::ystar:
        return _ystar_ions;

    case PeptideIon::bstar:
        return _bstar_ions;

    case PeptideIon::yo:
        return _yo_ions;

    case PeptideIon::bo:
        return _bo_ions;

    case PeptideIon::z:
        return _z_ions;

    case PeptideIon::a:
        return _a_ions;

    case PeptideIon::astar:
        return _astar_ions;
    case PeptideIon::ao:
        return _ao_ions;

    case PeptideIon::c:
        return _c_ions;
    case PeptideIon::x:
        return _x_ions;
    }
    return false;
}

Spectrum XtandemSpectrumProcess::process (const Spectrum & spectrum, mz parent_ion_mz, unsigned int parent_charge) const {
    qDebug() << "XtandemSpectrumProcess::process parent_charge==" << parent_charge;

    //1) clean isotopes
    Spectrum spectrum_process(spectrum);
    if (_remove_isotope) {
        spectrum_process = spectrum_process.xtandemDeisotope();
    }

    //2) remove parent ion mass
    if (_exclude_parent) {
        spectrum_process = spectrum_process.removeMassRange(MassRange(parent_ion_mz,Precision::getDaltonInstance(_exclude_parent_lower_dalton/parent_charge),Precision::getDaltonInstance(_exclude_parent_upper_dalton/parent_charge)));
    }
    //3) remove low masses
    //4) normalization
    spectrum_process = spectrum_process.removeMzUnder(_minimum_mz).takeNmostIntense(_nmost_intense).applyDynamicRange(_dynamic_range).removeIntensityUnder(1);
    //5) remove neutral loss
    if (_exclude_parent_neutral_loss) {
        pappso_double parent_ion_mhplus= ((parent_ion_mz - (MHPLUS*parent_charge))*parent_charge)+MHPLUS ;
        spectrum_process = spectrum_process.removeMassRange(MassRange(parent_ion_mhplus-_neutral_loss_mass,Precision::getDaltonInstance(_neutral_loss_window_dalton)));
    }
    //6) clean isotopes
    //7) keep n most intense peaks
    return spectrum_process.xtandemRemoveC13().takeNmostIntense(_nmost_intense);

}

}
