/**
 * \file pappsomspp/amino_acid/aamodification.h
 * \date 7/3/2015
 * \author Olivier Langella
 * \brief amino acid modification model
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include<QRegExp>
#include<QDebug>
#include <cmath>

#include "aamodification.h"
#include "../pappsoexception.h"
#include "../mass_range.h"
#include "../peptide/peptide.h"
#include "../obo/filterobopsimodsink.h"
#include "../obo/filterobopsimodtermaccession.h"
#include "../exception/exceptionnotfound.h"

/*

inline void initMyResource() {
    Q_INIT_RESOURCE(resources);
}
*/

namespace pappso {

QMutex AaModification::_mutex;

AaModification::AaModification(const QString & accession, pappso_double mass):_accession(accession), _mass(mass)
{
    _atom_count = {{AtomIsotopeSurvey::C, 0}, {AtomIsotopeSurvey::H, 0}, {AtomIsotopeSurvey::N, 0}, {AtomIsotopeSurvey::O, 0}, {AtomIsotopeSurvey::S, 0}};

    _map_isotope = {{Isotope::C13, 0},{Isotope::H2, 0},{Isotope::N15, 0},{Isotope::O17, 0},{Isotope::O18, 0},{Isotope::S33, 0},{Isotope::S34, 0},{Isotope::S36, 0}};

}


AaModification::AaModification(AaModification && toCopy) //move constructor
    :  _atom_count(std::move(toCopy._atom_count)), _accession(toCopy._accession), _mass(toCopy._mass), _map_isotope(toCopy._map_isotope)
{
}

AaModification::~AaModification()
{

}

AaModification::MapAccessionModifications AaModification::_map_accession_modifications = []
{
    MapAccessionModifications ret;

    return ret;
}();

AaModification * AaModification::createInstance(const OboPsiModTerm & term) {
    AaModification * new_mod;
    // qDebug() << " AaModification::createInstance begin";
    new_mod = new AaModification(term._accession, term._diff_mono);
    //xref: DiffFormula: "C 0 H 0 N 0 O 1 S 0"
    new_mod->setDiffFormula(term._diff_formula);
    new_mod->_name = term._name;
    return new_mod;
}

AaModification * AaModification::createInstance(const QString & accession) {
    if (accession == "internal:Nter_hydrolytic_cleavage_H") {
        OboPsiModTerm term;
        term._accession = accession;
        term._diff_formula = "H 1";
        term._diff_mono = MPROTIUM;
        term._name = "Nter hydrolytic cleavage H+";
        return (AaModification::createInstance(term));
    }
    if (accession == "internal:Cter_hydrolytic_cleavage_HO") {
        OboPsiModTerm term;
        term._accession = accession;
        term._diff_formula = "H 1 O 1";
        term._diff_mono = MPROTIUM + MASSOXYGEN;
        term._name = "Cter hydrolytic cleavage HO";
        return (AaModification::createInstance(term));
    }
    //initMyResource();
    FilterOboPsiModSink term_list;
    FilterOboPsiModTermAccession filter_accession(term_list, accession);

    OboPsiMod psimod(filter_accession);

    try {
        return (AaModification::createInstance(term_list.getOne()));
    }
    catch (ExceptionNotFound e) {
        throw ExceptionNotFound(QObject::tr("modification not found : [%1]\n%2").arg(accession).arg(e.qwhat()));
    }
}


void AaModification::setDiffFormula(const QString & diff_formula) {
    QRegExp rx("(^|\\s)([C,H,O,N,H,S])\\s([-]{0,1}\\d+)");
    int pos = 0;

    while ((pos = rx.indexIn(diff_formula, pos)) != -1) {
        qDebug() << rx.cap(2)<< " " << rx.cap(3);
        if (rx.cap(2) == "C") {
            _atom_count[AtomIsotopeSurvey::C] = rx.cap(3).toInt();
        } else if (rx.cap(2) == "H") {
            _atom_count[AtomIsotopeSurvey::H] = rx.cap(3).toInt();
        } else if (rx.cap(2) == "N") {
            _atom_count[AtomIsotopeSurvey::N] = rx.cap(3).toInt();
        } else if (rx.cap(2) == "O") {
            _atom_count[AtomIsotopeSurvey::O] = rx.cap(3).toInt();
        } else if (rx.cap(2) == "S") {
            _atom_count[AtomIsotopeSurvey::S] = rx.cap(3).toInt();
        }
        pos += rx.matchedLength();
    }

    //look for isotopes :
    rx.setPattern("\\(([-]{0,1}\\d+)\\)([C,H,O,N,H,S])\\s([-]{0,1}\\d+)");
    pos = 0;

    while ((pos = rx.indexIn(diff_formula, pos)) != -1) {
        qDebug() << rx.cap(1)<< " " << rx.cap(2)<< " " << rx.cap(3);
        int number_of_isotopes = rx.cap(3).toInt();
        if (rx.cap(2) == "C") {
            if (rx.cap(1) == "13") {
                _map_isotope.at(Isotope::C13) = number_of_isotopes;
            }
            _atom_count[AtomIsotopeSurvey::C] += number_of_isotopes;
        } else if (rx.cap(2) == "H") {
            if (rx.cap(1) == "2") {
                _map_isotope.at(Isotope::H2) = number_of_isotopes;
            }
            _atom_count[AtomIsotopeSurvey::H] += number_of_isotopes;
        } else if (rx.cap(2) == "N") {
            if (rx.cap(1) == "15") {
                _map_isotope.at(Isotope::N15) = number_of_isotopes;
            }
            _atom_count[AtomIsotopeSurvey::N] += number_of_isotopes;
        } else if (rx.cap(2) == "O") {
            if (rx.cap(1) == "17") {
                _map_isotope.at(Isotope::O17) = number_of_isotopes;
            }
            else if (rx.cap(1) == "18") {
                _map_isotope.at(Isotope::O18) = number_of_isotopes;
            }
            _atom_count[AtomIsotopeSurvey::O] += number_of_isotopes;
        } else if (rx.cap(2) == "S") {
            if (rx.cap(1) == "33") {
                _map_isotope.at(Isotope::S33) = number_of_isotopes;
            }
            else if (rx.cap(1) == "34") {
                _map_isotope.at(Isotope::S34) = number_of_isotopes;
            } else if (rx.cap(1) == "36") {
                _map_isotope.at(Isotope::S36) = number_of_isotopes;
            }
            _atom_count[AtomIsotopeSurvey::S] += number_of_isotopes;
        }
        pos += rx.matchedLength();
    }


    calculateMassFromChemicalComponents();

}


void AaModification::calculateMassFromChemicalComponents() {
    pappso_double theoretical_mass = 0;
    std::map<AtomIsotopeSurvey,int>::const_iterator it_atom =  _atom_count.find( AtomIsotopeSurvey::C );
    if (it_atom != _atom_count.end()) {
        theoretical_mass += MASSCARBON * (it_atom->second);
    }
    it_atom =  _atom_count.find( AtomIsotopeSurvey::H );
    if (it_atom != _atom_count.end()) {
        theoretical_mass += MPROTIUM * (it_atom->second);
    }

    it_atom =  _atom_count.find( AtomIsotopeSurvey::O );
    if (it_atom != _atom_count.end()) {
        theoretical_mass += MASSOXYGEN * (it_atom->second);
    }

    it_atom =  _atom_count.find( AtomIsotopeSurvey::N );
    if (it_atom != _atom_count.end()) {
        theoretical_mass += MASSNITROGEN * (it_atom->second);
    }
    it_atom =  _atom_count.find( AtomIsotopeSurvey::S );
    if (it_atom != _atom_count.end()) {
        theoretical_mass += MASSSULFUR * (it_atom->second);
    }

    qDebug() << theoretical_mass;

    theoretical_mass += DIFFC12C13 * _map_isotope.at(Isotope::C13);
    theoretical_mass += DIFFH1H2 * _map_isotope.at(Isotope::H2);
    theoretical_mass += DIFFN14N15 * _map_isotope.at(Isotope::N15);
    theoretical_mass += DIFFO16O17 * _map_isotope.at(Isotope::O17);
    theoretical_mass += DIFFO16O18 * _map_isotope.at(Isotope::O18);
    theoretical_mass += DIFFS32S33 * _map_isotope.at(Isotope::S33);
    theoretical_mass += DIFFS32S34 * _map_isotope.at(Isotope::S34);
    theoretical_mass += DIFFS32S36 * _map_isotope.at(Isotope::S36);


    pappso_double diff = std::fabs((pappso_double) _mass - theoretical_mass);
    if (diff < 0.001) {
        _mass = theoretical_mass;
        qDebug() << "AaModification::calculateMassFromChemicalComponents " << diff;
    }
    else {
        qDebug() << "ERROR in AaModification::calculateMassFromChemicalComponents theo="<< theoretical_mass << " m=" << _mass << " diff=" << diff;
    }

}

AaModificationP AaModification::getInstanceCustomizedMod(mz modificationMass) {
    QString accession = QString("%1").arg(modificationMass);
    qDebug() << "AaModification::getInstanceCustomizedMod " << accession;
    if (_map_accession_modifications.find(accession) == _map_accession_modifications.end() ) {
        QMutexLocker locker(&_mutex);
        // not found
        _map_accession_modifications.insert(
            std::pair<QString, AaModification* >(accession,new AaModification(accession, modificationMass)));
    } else {
        // found
    }
    return _map_accession_modifications.at(accession);
}

AaModificationP AaModification::getInstance(const QString & accession) {
    try {
        MapAccessionModifications::iterator it = _map_accession_modifications.find(accession);
        if (it == _map_accession_modifications.end() ) {

            QMutexLocker locker(&_mutex);
            // not found
            std::pair< MapAccessionModifications::iterator, bool > insert_res = _map_accession_modifications.insert(
                        std::pair<QString, AaModification* >(accession, AaModification::createInstance(accession)));
            it = insert_res.first;
        } else {
            // found
        }
        return it->second;
    }
    catch (ExceptionNotFound e) {
        throw ExceptionNotFound(QObject::tr("ERROR getting instance of : %1 NOT FOUND\n%2").arg(accession).arg(e.qwhat()));
    }
    catch (PappsoException e) {
        throw PappsoException(QObject::tr("ERROR getting instance of %1\n%2").arg(accession).arg(e.qwhat()));
    }
    catch (std::exception e) {
        throw PappsoException(QObject::tr("ERROR getting instance of %1\n%2").arg(accession).arg(e.what()));
    }
}

AaModificationP AaModification::getInstance(const OboPsiModTerm & oboterm) {

    MapAccessionModifications::iterator it = _map_accession_modifications.find(oboterm._accession);
    if (it == _map_accession_modifications.end() ) {
        QMutexLocker locker(&_mutex);
        // not found
        std::pair< MapAccessionModifications::iterator, bool > insert_res = _map_accession_modifications.insert(
                    std::pair<QString, AaModification* >(oboterm._accession, AaModification::createInstance(oboterm)));
        it = insert_res.first;
    } else {
        // found
    }
    return it->second;
}


AaModificationP AaModification::getInstanceXtandemMod(const QString & type,pappso_double mass, const PeptideSp & peptide_sp, unsigned int position) {
    PrecisionP precision = Precision::getDaltonInstance(0.001);
    if (MassRange(mass, precision).contains(getInstance("MOD:00719")->getMass())) {
        if (type == "M") {
            return getInstance("MOD:00719");
        }
        if (type == "K") {
            return getInstance("MOD:01047");
        }
    }
    //accession== "MOD:00057"
    if (MassRange(mass, precision).contains(getInstance("MOD:00408")->getMass())) {
//id: MOD:00394
//name: acetylated residue
        //potential N-terminus modifications
        if (position == 0) {
            return getInstance("MOD:00408");
        }
    }
    if (MassRange(mass, precision).contains(getInstance("MOD:01160")->getMass())) {
        //-17.02655
        //loss of ammonia [MOD:01160] -17.026549
        return getInstance("MOD:01160");
    }

    if (MassRange(mass, precision).contains(getInstance("MOD:01060")->getMass())) {
        //// iodoacetamide [MOD:00397] 57.021464
        if (type == "C") {
            return getInstance("MOD:01060");
        } else {
            return getInstance("MOD:00397");
        }
    }
    if (MassRange(mass, precision).contains(getInstance("MOD:00704")->getMass())) {
        //loss of water
        /*
          if (position == 0) {
              if (peptide_sp.get()->getSequence().startsWith("EG")) {
                  return getInstance("MOD:00365");
              }
              if (peptide_sp.get()->getSequence().startsWith("ES")) {
                  return getInstance("MOD:00953");
              }
              if (type == "E") {
                  return getInstance("MOD:00420");
              }
          }
        */
        // dehydrated residue [MOD:00704] -18.010565
        return getInstance("MOD:00704");
    }
    if (MassRange(mass, precision).contains(getInstance("MOD:00696")->getMass())) {
        //phosphorylated residue [MOD:00696] 79.966330
        return getInstance("MOD:00696");
    }
    bool isCter = false;
    if (peptide_sp.get()->size() == (position+1)) {
        isCter = true;
    }
    if ((position == 0) || isCter) {
        if (MassRange(mass, precision).contains(getInstance("MOD:00429")->getMass())) {
            //dimethyl
            return getInstance("MOD:00429");
        }
        if (MassRange(mass, precision).contains(getInstance("MOD:00552")->getMass())) {
            //4x(2)H labeled dimethyl residue
            return getInstance("MOD:00552");
        }
        if (MassRange(mass, precision).contains(getInstance("MOD:00638")->getMass())) {
            //2x(13)C,6x(2)H-dimethylated arginine
            return getInstance("MOD:00638");
        }
    }
    throw PappsoException(QObject::tr("tandem modification not found : %1 %2 %3 %4").arg(type).arg(mass).arg(peptide_sp.get()->getSequence()).arg(position));
}

pappso_double AaModification::getMass() const {
    return _mass;
}


int AaModification::getNumberOfAtom(AtomIsotopeSurvey atom) const {
    //qDebug() << "AaModification::getNumberOfAtom(AtomIsotopeSurvey atom) NOT IMPLEMENTED";
    return _atom_count.at(atom);
}



int AaModification::getNumberOfIsotope(Isotope isotope) const {
    try {
        return _map_isotope.at(isotope);
    }
    catch (std::exception e) {
        throw PappsoException(QObject::tr("ERROR in AaModification::getNumberOfIsotope %2").arg(e.what()));
    }
}


bool AaModification::isInternal() const {
    if (_accession.startsWith("internal:")) {
        return true;
    }
    return false;
}
}


