/**
 * \file pappsomspp/types.h
 * \date 4/3/2015
 * \author Olivier Langella
 * \brief This header contains all the type re-definitions and all
 * the global variables definitions used in the PAPPSOms++ library.
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#ifndef _PAPPSOMSPP_TYPES_H_
#define _PAPPSOMSPP_TYPES_H_ 1

#include <QString>

namespace pappso {
/************ Typedefs **************************************************/

/** \var typedef QString PeptideStr
 \brief A type definition for PeptideStr
 */
typedef QString PeptideStr;

/** \var typedef double pappso_double
 \brief A type definition for doubles
 */
typedef double pappso_double;

/** \var typedef float mcq_float
 \brief A type definition for floats
 */
typedef float pappso_float;

/** \def specific type for an mz mass
 *
 */
typedef pappso_double mz;

/*********** enumerations *********************************/

/** \def PrecisionUnit ppm or dalton
 *
 */
enum class PrecisionUnit {
    ppm, dalton
};

/** \def AtomIsotopeSurvey list of atoms on which isotopes may occurs
 *
 */
enum class AtomIsotopeSurvey {
    C, H, O, N, S
};


/** \def Isotope list of isotopes taken into account for peptide abundance calculation
 *
 */
enum class Isotope {
    C13, H2, O17, O18, N15, S33, S34, S36
};


/*********** Global variables definitions*********************************/

/** \def MHPLUS 1.007276466879
 \brief The (monoisotopic) mass of the H+ ion
 https://en.wikipedia.org/wiki/Proton (One Proton alone)
 1.007276466879
 */
const pappso_double MHPLUS(1.007276466879);
/** \def MPROTIUM 1.00782503207
 \brief The (monoisotopic) mass of the H atom
 https://en.wikipedia.org/wiki/Isotopes_of_hydrogen (One proton + One electron)
 1.00782503207
 */
const pappso_double MPROTIUM(1.00782503207);



/** \def ONEMILLION 1000000
 \brief One million integer, why not.
 */
const pappso_double ONEMILLION(1000000);


const pappso_double MASSOXYGEN(15.99491461956);
const pappso_double MASSCARBON(12);
const pappso_double MASSH2O((MPROTIUM * 2) + MASSOXYGEN);
const pappso_double MASSNITROGEN(14.0030740048);
const pappso_double MASSNH3((MPROTIUM * 3) + MASSNITROGEN);
const pappso_double MASSCO(MASSCARBON + MASSOXYGEN);
const pappso_double MASSPHOSPHORUS(30.973761998);
const pappso_double MASSSULFUR(31.97207100);

//id: MOD:00696 name: phosphorylated residue H 1 O 3 P 1
const pappso_double MASSPHOSPHORYLATEDR(MPROTIUM + (MASSOXYGEN*3) + MASSPHOSPHORUS);


//CHNOS

/** \def DIFFC12C13 1.0033548378
 \brief The (monoisotopic) mass difference between C12 (12u) and C13 stable isotope of carbon
 */
const pappso_double DIFFC12C13(1.0033548378);

/** \def DIFFS32S33 0.99938776
 \brief The (monoisotopic) mass difference between S32 (31.97207100u) and S33 (32.97145876u) stable isotope of sulfur
 https://en.wikipedia.org/wiki/Isotopes_of_sulfur
 */
const pappso_double DIFFS32S33(32.97145876 - MASSSULFUR);

/** \def DIFFS32S34 1.9957959
 \brief The (monoisotopic) mass difference between S32 (31.97207100u) and S34 (33.96786690u) stable isotope of sulfur
 */
const pappso_double DIFFS32S34(33.96786690 - MASSSULFUR);

/** \def DIFFS32S36 3.99500976
 \brief The (monoisotopic) mass difference between S32 (31.97207100u) and S36 (35.96708076u) stable isotope of sulfur
 */
const pappso_double DIFFS32S36(35.96708076 - MASSSULFUR);


/** \def DIFFH1H2
 \brief The (monoisotopic) mass difference between H1 and H2 stable isotope of hydrogen
 */
const pappso_double DIFFH1H2(2.0141017778 - MPROTIUM);

/** \def DIFFO16O18
 \brief The (monoisotopic) mass difference between O16 and O18 stable isotope of oxygen
 */
const pappso_double DIFFO16O18(17.9991610 - MASSOXYGEN);

/** \def DIFFO16O17
 \brief The (monoisotopic) mass difference between O16 and O17 stable isotope of oxygen
 */
const pappso_double DIFFO16O17(16.99913150 - MASSOXYGEN);

/** \def DIFFN14N15
 \brief The (monoisotopic) mass difference between N14 and N15 stable isotope of nitrogen
 */
const pappso_double DIFFN14N15(15.0001088982 - MASSNITROGEN);


//http://education.expasy.org/student_projects/isotopident/htdocs/motza.html
/** \def ABUNDANCEH2 0.0156%
 \brief H2 isotope abundance
 */
const pappso_double ABUNDANCEH2(0.00015574);

/** \def ABUNDANCEN15 0.00364
 \brief N15 isotope abundance
 */
const pappso_double ABUNDANCEN15(0.003663);

/** \def ABUNDANCEO17
 \brief O17 isotope abundance
 */
const pappso_double ABUNDANCEO17(0.000372);

/** \def ABUNDANCEO18 0.2%
 \brief O18 isotope abundance
 */
const pappso_double ABUNDANCEO18(0.0020004);

/** \def ABUNDANCEC13 1.109%
 \brief C13 isotope abundance
 */
const pappso_double ABUNDANCEC13(0.011078);

/** \def ABUNDANCEC12 98.89%
 \brief C12 abundance
 */
const pappso_double ABUNDANCEC12(0.988922);

/** \def ABUNDANCES33 0.00750
 \brief S33 abundance
 */
const pappso_double ABUNDANCES33(0.00750);

/** \def ABUNDANCES34 0.0429
 \brief S34 abundance
 */
const pappso_double ABUNDANCES34(0.04215);

/** \def ABUNDANCES36 0.00020
 \brief S36 abundance
 */
const pappso_double ABUNDANCES36(0.00020);





}

#endif /* _PAPPSOMSPP_TYPES_H_ */
