/**
 * \file pappsomspp/spectrum/spectrum.cpp
 * \date 15/3/2015
 * \author Olivier Langella
 * \brief basic spectrum
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include <QDebug>
#include "spectrum.h"
#include "../mass_range.h"
#include "../pappsoexception.h"

#include <cmath>
#include <limits>
#include <vector>
#include <QDebug>
#include <map>

#include "../peptide/peptidefragmentionlistbase.h"

#include "../exception/exceptionoutofrange.h"

namespace pappso {


static bool operator_minus_intensity(const Peak & a, const Peak & b)
{
    return (a.intensity < b.intensity);
}


bool sort_vpeaks_byintensity (Peak i,Peak j) {
    return (i.intensity >j.intensity);
}
bool sort_vpeaks_bymz (Peak i,Peak j) {
    return (i.mz <j.mz);
}


Spectrum::Spectrum()  {
    _v_peaks.resize(0);
}

Spectrum::Spectrum(std::vector<pair<pappso_double, pappso_double>> & v_in) {
    _v_peaks.reserve(v_in.size());
    for (auto && pairin : v_in) {
        _v_peaks.push_back(Peak(pairin));
    }
    std::sort (_v_peaks.begin(), _v_peaks.end(), sort_vpeaks_bymz);

}

Spectrum::Spectrum(const Spectrum & toCopy) :
    _v_peaks(toCopy._v_peaks) {
    qDebug() << "copy Spectrum::Spectrum(const Spectrum & toCopy)";
    //qDebug() << "***********************************************************";
    // qDebug() << "Spectrum::Spectrum copy constructor";
    // qDebug() << "Spectrum::Spectrum copy constructor";
    // qDebug() << "***********************************************************";
}

Spectrum::Spectrum(Spectrum && toCopy) : _v_peaks(std::move(toCopy._v_peaks)) {
    qDebug() << "move Spectrum::Spectrum(Spectrum && toCopy)";
}

Spectrum& Spectrum::operator=(const Spectrum& other) {
    _v_peaks = other._v_peaks;
    qDebug() << "copy assigned  Spectrum::operator=" ;
    return *this;
}

Spectrum& Spectrum::operator=(Spectrum&& toCopy) {
    qDebug() << "move Spectrum::operator=(Spectrum&& toCopy)";
    _v_peaks = std::move(toCopy._v_peaks);
    return *this;
}


SpectrumSp Spectrum::makeSpectrumSp() const {
    return std::make_shared<const Spectrum>(*this);
}

Spectrum::~Spectrum() {
    _v_peaks.clear();
}

void Spectrum::debugPrintValues() const {
    for (auto && peak : _v_peaks) {
        qDebug() << "mz = " << peak.mz << ", int = " << peak.intensity;
    }
}

const unsigned int Spectrum::getSpectrumSize() const {
    return (_v_peaks.size());
}

void Spectrum::clear() {
    _v_peaks.clear();
}

void Spectrum::reserve(const unsigned int size) {
    _v_peaks.reserve(size);
}



const Peak Spectrum::getMaxIntensity() const {
    //pappso_double max(numeric_limits<pappso_double>::quiet_NaN());

    auto it  = std::max_element(_v_peaks.begin(), _v_peaks.end(), operator_minus_intensity) ;

    if (it == _v_peaks.end()) {
        throw ExceptionOutOfRange(QObject::tr("unable to get max peak intensity on spectrum size %1").arg(_v_peaks.size()));
    }
    return (*it);
}

void Spectrum::push_back(const Peak& mz_int_pair) {
    _v_peaks.push_back(mz_int_pair);
}

/** @brief take n most intense peaks from the spectrum to copy
 *
 * @param n
 * @param toCopy
 */
Spectrum Spectrum::takeNmostIntense(unsigned int n) const {
    auto v_peaks_by_intensity(_v_peaks);
    std::sort (v_peaks_by_intensity.begin(), v_peaks_by_intensity.end(), sort_vpeaks_byintensity);

    Spectrum new_spectrum;
    for (auto && peak : v_peaks_by_intensity) {
        new_spectrum.push_back(peak);
        if (new_spectrum._v_peaks.size() == n) break;
    }
    //new_spectrum.debugPrintValues();
    std::sort (new_spectrum._v_peaks.begin(), new_spectrum._v_peaks.end(), sort_vpeaks_bymz);

    return new_spectrum;
}

Spectrum Spectrum::applyDynamicRange(pappso_double dynamic_range) const {

    Peak max_int = getMaxIntensity();
    Spectrum new_spectrum;
    for (auto && peak : _v_peaks) {
        new_spectrum.push_back(Peak(peak.mz, (peak.intensity / max_int.intensity) * dynamic_range));
    }
    return new_spectrum;
}

Spectrum Spectrum::filterSum(const MassRange & mass_range) const {
    Spectrum new_spectrum;
    std::vector<Peak>::const_iterator it = _v_peaks.begin();
    std::vector<Peak>::const_reverse_iterator itrev = _v_peaks.rbegin();
    std::vector<Peak>::const_iterator itend = _v_peaks.end();
    std::vector<Peak>::const_reverse_iterator itrevend = _v_peaks.rend();
    
    pappso_double lower_bound = mass_range.getLowest();
    pappso_double upper_bound = mass_range.getHighest();
    
    
    while ((it != itend) &&(it->mz <= itrev->mz) && (itrev != itrevend)) {
        pappso::mz sum = it->mz + itrev->mz;
        if (sum < lower_bound) {
            it++;
        }
        else if (sum > upper_bound) {
            itrev++;
        }
        else {
            new_spectrum.push_back(*it);
            new_spectrum.push_back(*itrev);
            std::vector<Peak>::const_reverse_iterator itrevin = itrev;
            itrevin++;
            sum = it->mz + itrevin->mz;
            while ((sum > lower_bound) &&(it->mz <= itrevin->mz) && (itrevin != itrevend)) {
                sum = it->mz + itrevin->mz;
                //new_spectrum.push_back(*it);
                new_spectrum.push_back(*itrevin);
                itrevin++;
            }
            it++;
        }
    }
    std::sort (new_spectrum._v_peaks.begin(), new_spectrum._v_peaks.end(), sort_vpeaks_bymz);
    std::vector<Peak>::iterator itendfix = std::unique(new_spectrum._v_peaks.begin(), new_spectrum._v_peaks.end(), [](const Peak & l, const Peak & r){
        return l.mz == r.mz && l.intensity == r.intensity;
    });
    new_spectrum._v_peaks = std::vector<Peak>(new_spectrum._v_peaks.begin(),itendfix );
    return new_spectrum;
}

Spectrum Spectrum::removeMassRange(const MassRange & remove_mass_range) const {
    pappso_double lower_bound = remove_mass_range.getLowest();
    pappso_double upper_bound = remove_mass_range.getHighest();
    Spectrum new_spectrum;
    for (auto && peak : _v_peaks) {
        if (peak.mz < lower_bound) {
            new_spectrum.push_back(peak);
        }
        else  if (peak.mz > upper_bound) {
            new_spectrum.push_back(peak);
        }
    }
    return new_spectrum;

}


Spectrum Spectrum::round() const {
    Spectrum new_spectrum;
    for (auto&& peak : _v_peaks) {
        new_spectrum.push_back(Peak(peak.mz, std::round(peak.intensity)));
    }
    return new_spectrum;
}

Spectrum Spectrum::floor() const {
    Spectrum new_spectrum;
    for (auto&& peak : _v_peaks) {
        new_spectrum.push_back(Peak(peak.mz, std::floor(peak.intensity)));
    }
    return new_spectrum;
}

/** @brief remove small peaks from less than 0.95daltons of an other
 *
 */
Spectrum Spectrum::xtandemDeisotope() const {

    Spectrum new_spectrum;
    if (_v_peaks.size() > 0) {
        auto itpeak = _v_peaks.begin()+1;
        auto itpeakend = _v_peaks.end();
        Peak peak_to_write = _v_peaks.at(0);
        while (itpeak != itpeakend) {
            if ((itpeak->mz < 200) || ((itpeak->mz - peak_to_write.mz)>=0.95)) {
                new_spectrum.push_back(peak_to_write);
                peak_to_write = *itpeak;
            }
            else if (itpeak->intensity > peak_to_write.intensity) {
                peak_to_write = *itpeak;
            }
            itpeak++;
        }
    }
    return new_spectrum;
}

Spectrum Spectrum::xtandemRemoveC13() const {

    Spectrum new_spectrum;
    if (_v_peaks.size() > 0) {
        auto itpeak = _v_peaks.begin();
        auto itpeaknext = _v_peaks.begin()+1;
        auto itpeakend = _v_peaks.end();

        pappso::mz old_mz = itpeak->mz;

        while( itpeaknext != itpeakend )
        {
            if( ( itpeaknext->mz - old_mz) >= 1.5  || itpeaknext->mz < 200.0)
            {
                new_spectrum.push_back(*itpeak);
                itpeak = itpeaknext;
                old_mz = itpeak->mz;
            }
            else if( itpeaknext->intensity > itpeak->intensity )
            {
                itpeak = itpeaknext;
            }
            itpeaknext++;
        }
        new_spectrum.push_back(*itpeak);
    }
    return new_spectrum;
}

Spectrum Spectrum::removeMzUnder(pappso_double mz_min) const {
    Spectrum new_spectrum;
    for (auto && peak : _v_peaks) {
        if (mz_min <= peak.mz) {
            new_spectrum.push_back(peak);
        }
    }
    return new_spectrum;
}

Spectrum Spectrum::removeIntensityUnder(pappso_double minimum_intensity) const {
    Spectrum new_spectrum;
    for (auto && peak : _v_peaks) {
        if (minimum_intensity <= peak.intensity) {
            new_spectrum.push_back(peak);
        }
    }
    return new_spectrum;
}

Spectrum Spectrum::removeIntensityPercentUnder(float percentIntMIn) const {
    pappso_double max = this->getMaxIntensity().intensity;
    pappso_double intCutOff = max * (pappso_double) percentIntMIn;
    return this->removeIntensityUnder(intCutOff);
}


bool Spectrum::equals(const Spectrum & other, PrecisionP precision) const {
    if (size() != other.size()) {
        qDebug()<< size() << " vs size other " << other.size();
        return false;
    }
    PrecisionP precint = Precision::getPpmInstance(10);
    auto other_peak_it = other._v_peaks.begin();
    for (auto && peak : _v_peaks) {
        qDebug() << "first " << peak.mz << " " << peak.intensity << " second " << other_peak_it->mz << " " << other_peak_it->intensity;
        if (!MassRange(peak.mz, precision).contains(other_peak_it->mz)) {
            qDebug() << "mz " << peak.mz << " != " << other_peak_it->mz;
            return false;
        }
        if (!MassRange(peak.intensity, precint).contains(other_peak_it->intensity) ) {
            qDebug() << "intensity " << peak.intensity << " != " << other_peak_it->intensity;
            return false;
        }
        other_peak_it++;
    }
    return true;
}
}
