/**
 * \file pappsomspp/mass_range.cpp
 * \date 4/3/2015
 * \author Olivier Langella
 * \brief object to handle a mass range (an mz value + or - some delta)
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and implementation
 ******************************************************************************/

#include "mass_range.h"
#include <cmath>
#include <QDebug>

using namespace pappso;


Precision::MapPpmPrecision Precision::_map_ppm = []
{
    MapPpmPrecision ret;

    return ret;
}();


Precision::MapDaltonPrecision Precision::_map_dalton = []
{
    MapDaltonPrecision ret;

    return ret;
}();

PrecisionP Precision::getDaltonInstance(pappso_double precision) {
    MapDaltonPrecision::iterator it = Precision::_map_dalton.find(precision);
    if (it == Precision::_map_dalton.end() ) {
        // not found
        std::pair< MapDaltonPrecision::iterator, bool > insert_res = Precision::_map_dalton.insert(
                    std::pair<pappso_double, DaltonPrecision * > (precision, new DaltonPrecision(precision)));
        it = insert_res.first;
    } else {
        // found
    }
    return it->second;
}
PrecisionP Precision::getPpmInstance(pappso_double precision) {
    MapPpmPrecision::iterator it = Precision::_map_ppm.find(precision);
    if (it == Precision::_map_ppm.end() ) {
        // not found
        std::pair< MapPpmPrecision::iterator, bool > insert_res = Precision::_map_ppm.insert(
                    std::pair<pappso_double, PpmPrecision * >(precision, new PpmPrecision(precision)));
        it = insert_res.first;
    } else {
        // found
    }
    return it->second;
}


MassRange::MassRange(mz mz, PrecisionP precision): _mz(mz),_delta(precision->getDelta(_mz)) {
}


MassRange::MassRange(pappso::mz mz, PrecisionP precision_lower, PrecisionP precision_upper) {
  
  _delta = (precision_lower->getDelta(mz) + precision_upper->getDelta(mz))/2;
  _mz = mz - precision_lower->getDelta(mz)  + _delta;
}
    
    
MassRange::MassRange (const MassRange & other): _mz(other._mz),_delta(other._delta) {
     //std::cout << "MassRange::MassRange (const MassRange & other)" << endl;
}

const mz MassRange::getMz() const {
    return _mz;
}

bool MassRange::contains(mz mz) const {
    //qDebug() << " " << std::abs(mz - _mz)  << " _delta:" << _delta;
    if (std::abs(mz - _mz) <= _delta) {
        return true;
    }
    return false;
}

const QString MassRange::toString() const {
    //QString s = "mz=" + QString::number(_mz) + " delta=" + QString::number(_delta);
    return QString("mz=%1 delta=%2 : %3 < %4 < %5").arg(_mz).arg(_delta).arg(getLowest()).arg(_mz).arg(getHighest());
}
