/*
    libodsstream is a library to read and write ODS documents as streams
    Copyright (C) 2013  Olivier Langella <Olivier.Langella@moulon.inra.fr>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "odsdocwriter.h"
#include "structure/manifestxml.h"
#include "structure/metaxml.h"
#include "structure/stylesxml.h"

#include <quazip/quazipfile.h>
#include <quazip/quacrc32.h>

OdsDocWriter::OdsDocWriter(QIODevice * device) {
    _p_quaZip = new QuaZip(device);
    _p_quaZip->open(QuaZip::mdCreate);

    //_p_quaZip->setComment("application/vnd.oasis.opendocument.spreadsheet");
    QByteArray* messageIn = new QByteArray();
    *messageIn = "application/vnd.oasis.opendocument.spreadsheet";

    QuaZipFile outFile(_p_quaZip);
    QuaCrc32 crc32;
    quint32  crc(crc32.calculate( *messageIn));
    QuaZipNewInfo info("mimetype");
    info.uncompressedSize = messageIn->length();
    outFile.open(QIODevice::WriteOnly,info,NULL,crc,0,0,true);
    outFile.write(*messageIn);
    outFile.close();

    delete (messageIn);

    ManifestXml manifest(_p_quaZip);
    MetaXml metaxml(_p_quaZip);
    _p_content = new ContentXml(_p_quaZip);

}

/**
* ends the document creation. It is required to obtain a correct output
*
*/
void OdsDocWriter::close() {
    _p_content->close();
    delete _p_content;
    StylesXml stylesXml(_p_quaZip);
    _p_quaZip->close();
    delete _p_quaZip;
    //zipOutputStream = null;
}


/**
 * adds a new datasheet to the document
 *
 * @param sheetName
 * @throws XMLStreamException
 */
void OdsDocWriter::writeSheet(const QString & sheetName) {
    _p_content->writeSheet(sheetName);
}

/**
 * adds a new line to the document
 *
 * @throws XMLStreamException
 */
void OdsDocWriter::writeLine() {
    //this.content.writeLine();
}

/**
 * adds a new text cell to the document
 *
 * @param text
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(const QString & text) {
    _p_content->writeCell(text);
//     if (text == null) {
//         this.writeEmptyCell();
//         return;
//     }
//     this.content.writeCell(text);
}

void OdsDocWriter::writeCell(const char * text) {
    QString textQt(text);
    _p_content->writeCell(textQt);
//     if (text == null) {
//         this.writeEmptyCell();
//         return;
//     }
//     this.content.writeCell(text);
}

/**
 * writes an empty cell
 *
 * @throws XMLStreamException
 */
void OdsDocWriter::writeEmptyCell() {
    _p_content->writeCell("");
    // this.content.writeCell("");
}

/**
 * adds a new cell containing an integer
 *
 * @param value
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(int value) {
    _p_content->writeCell(value);
    // this.content.writeCell(value);
}

/**
 * adds a new cell containing a float
 *
 * @param value
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(float value) {
    _p_content->writeCell(value);
    // this.content.writeCell(value);
}

/**
 * adds a new cell containing a double
 *
 * @param value
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(double value) {
    _p_content->writeCell(value);
    // this.content.writeCell(value);
}

/**
 * adds a new cell containing a boolean
 *
 * @param value
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(bool value) {
    _p_content->writeCell(value);
    //this.content.writeCell(value);
}

/**
 * adds a new cell containing a date
 *
 * @param date
 * @throws XMLStreamException
 * @throws DatatypeConfigurationException
 */
void OdsDocWriter::writeCell(const QDate & date)  {
    _p_content->writeCell(date);
//     if (date == null) {
//         this.writeEmptyCell();
//         return;
//     }
//     this.content.writeCell(date);
}

/**
 * adds a new cell containing a timestamp
 *
 * @param date
 * @throws XMLStreamException
 * @throws DatatypeConfigurationException
 */
void OdsDocWriter::writeCell(const QDateTime & date) {
    _p_content->writeCell(date);
//     if (date == null) {
//         this.writeEmptyCell();
//         return;
//     }
//     this.content.writeCell(new Date(date.getTime()));
}

/**
 * NOT WORKING, adds a new cell containing a duration
 *
 * @param duration
 * @throws XMLStreamException
 * @throws DatatypeConfigurationException
 */
// void OdsDocWriter::writeCell(Duration duration)  {
//     if (duration == null) {
//         this.writeEmptyCell();
//         return;
//     }
//     this.content.writeCell(duration);
// }

/**
 * adds a new cell containing a text inside an href link
 *
 * @param theUri
 * @param text
 * @throws XMLStreamException
 */
void OdsDocWriter::writeCell(const QUrl & theUri, const QString & text)  {
    _p_content->writeCell(theUri, text);
}
