/**
 * \file tests/test_spectralalignment.cpp
 * \date 07/11/2023
 * \author Olivier Langella
 * \brief tests for spectra objects
 */


/*
 * SpecGlobTool, Spectra to peptide alignment tool
 * Copyright (C) 2023  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
// ./tests/catch2-only-tests [align] -s


#include <QDebug>
#include <QString>
#include <QTextStream>
#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>
#include "config.h"
#include "common.h"
#include <pappsomspp/core/massspectrum/massspectrum.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include <pappsomspp/core/processing/filters/filterpass.h>
#include <pappsomspp/core/processing/specglob/spectralalignment.h>
#include <pappsomspp/core/utils.h>


/**
 * Show the 3 matrices : Score, Origin, E
 */
void
showMatrices(QTextStream &output_stream,
             const matrix<pappso::specglob::SpectralAlignmentDataPoint> &matrix)
{
  // output_stream << "\n" << getExpeSpec().getMassList());
  for(auto itr1 = matrix.begin1(); itr1 != matrix.end1(); ++itr1)
    {
      for(auto itr2 = itr1.begin(); itr2 != itr1.end(); itr2++)
        {
          output_stream << (*itr2).score << " ";
        }
      output_stream << Qt::endl;
    }

  output_stream << Qt::endl;
  for(auto itr1 = matrix.begin1(); itr1 != matrix.end1(); ++itr1)
    {
      for(auto itr2 = itr1.begin(); itr2 != itr1.end(); itr2++)
        {
          output_stream << QString("[%1,%2]")
                             .arg(Utils::toString((*itr2).alignment_type))
                             .arg((*itr2).origin_column_indices)
                        << " ";
        }
      output_stream << Qt::endl;
    }


  output_stream << Qt::endl;
  for(auto itr1 = matrix.begin1(); itr1 != matrix.end1(); ++itr1)
    {
      for(auto itr2 = itr1.begin(); itr2 != itr1.end(); itr2++)
        {
          output_stream << (*itr2).mass_difference << " ";
        }
      output_stream << Qt::endl;
    }
}


TEST_CASE("specglob test spectral alignment", "[align]")
{
  // Set the debugging message formatting pattern.scores to apply in comparisons
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: mzIdentML parser init ::..", "[align]")
  {

    pappso::PeptideSp sampler(
      std::make_shared<pappso::Peptide>("AIADGSLLDLLR"));

    pappso::specglob::PeptideSpectraCsp peptide_spectra =
      std::make_shared<pappso::specglob::PeptideSpectrum>(sampler);

    QualifiedMassSpectrum spectrum_simple =
      readMgf(QString(CMAKE_SOURCE_DIR)
                .append("/tests/data/peaklists/peaklist_15046.mgf"));
    pappso::FilterGreatestY(60).filter(
      *(spectrum_simple.getMassSpectrumSPtr().get()));

    PrecisionPtr precision_ptr =
      pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::specglob::ExperimentalSpectrumCsp experimental_spectrum =
      std::make_shared<pappso::specglob::ExperimentalSpectrum>(spectrum_simple,
                                                               precision_ptr);
    pappso::specglob::SpectralAlignment spectral_alignment(
      pappso::specglob::ScoreValues(), precision_ptr);

    spectral_alignment.align(peptide_spectra, experimental_spectrum);


    WARN(experimental_spectrum.get()->toString().toStdString());

    QTextStream ts(stdout);
    showMatrices(ts, spectral_alignment.getMatrix());
    std::vector<int> row0(
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0});
    REQUIRE(spectral_alignment.getScoreRow(0).size() == row0.size());
    REQUIRE_THAT(spectral_alignment.getScoreRow(0),
                 Catch::Matchers::Approx(row0));
    std::vector<int> row1(
      {0,  -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
       7,  7,  -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, 7,  -4, -4,
       7,  7,  7,  -4, -4, -4, -4, -4, 7,  -4, -4, -4, -4, -4, -4, -4, -4,
       -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
       -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, 7,  -4, -4, -4, -4,
       -4, -4, -4, -4, -4, -4, 10, 7,  -4, -4, -4, -4, -4, -4, -4, 7,  -4,
       -4, -4, -4, -4, -4, 7,  -4, -4, -4, -4, -4, -4, -4, 7,  7,  -4});
    REQUIRE_THAT(spectral_alignment.getScoreRow(1),
                 Catch::Matchers::Approx(row1));
    std::vector<int> row2_java(
      {0,  -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8,
       3,  3,  -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, 3,  -8, -8,
       3,  3,  3,  14, -1, -8, -8, -8, 14, -8, -8, -1, -8, -8, -8, -8, -8,
       -8, -8, -8, -8, -8, -8, -1, -1, -1, -1, -8, -1, -1, -1, -1, -8, -8,
       -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -1, 3,  -8, -8, -1, -8,
       -8, -8, -8, -8, -8, -8, 6,  3,  -8, -1, -8, -8, -8, -8, -8, 14, -8,
       -8, -8, -8, -8, -8, 3,  -8, -8, -8, -8, -1, -8, -8, 3,  3,  -8});


    std::vector<int> row2(
      {0,  -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8,
       3,  3,  -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, 3,  -8, -8,
       3,  3,  3,  14, -1, -8, -8, -8, 14, -8, -8, -1, -8, -8, -8, -8, -8,
       -8, -8, -8, -8, -8, -8, -1, -1, -1, -1, -8, -1, -1, -1, -1, -8, -8,
       -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -8, -1, 3,  -8, -8, -1, -8,
       -8, -8, -8, -8, -8, -8, 6,  3,  -8, -1, -8, -8, -8, -8, -8, 14, -8,
       -8, -8, -8, -8, -8, 3,  -8, -8, -8, -8, -1, -8, -8, 3,  3,  -8});
    REQUIRE_THAT(spectral_alignment.getScoreRow(2),
                 Catch::Matchers::Approx(row2_java));

    std::vector<int> row10(
      {0,   -40, -40, -40, -40, -40, -40, -40, -40, -40, -40, -40, -40, -40,
       -40, -40, -40, -28, -36, -28, -40, -40, -40, -40, -40, -40, -40, -40,
       -28, -40, -40, -36, -40, -8,  -33, -33, -33, -8,  -8,  -40, -40, -5,
       -8,  -12, -12, -8,  -40, -40, -11, -11, -40, -40, 0,   0,   2,   -12,
       -40, -4,  -4,  13,  -1,  -40, -2,  -2,  -2,  -2,  -40, -40, -11, -11,
       -40, 33,  33,  -11, -11, -40, -40, -40, -40, 44,  -22, 18,  18,  29,
       18,  -40, -40, -40, -40, -40, -40, -20, -11, -40, 29,  -40, -40, -40,
       -40, -40, 29,  -40, -40, -40, -40, -40, -40, -22, -11, -40, -40, -40,
       29,  -11, -40, -22, -7,  -40});
    // 0  -40  -40  -40  -40  -40  -40  -40  -40  -40  -40  -40  -40  -40  -40
    // -40  -40  -28  -36  -28  -40  -40  -40  -40  -40  -40  -40  -40  -28  -40
    // -40  -36  -40   -8  -33  -33  -33   -8   -8  -40  -40   -5   -8  -12  -12
    // -8  -40  -40  -11  -11  -40  -40    0    0    2  -12  -40   -4   -4   13
    // -1  -40   -2   -2   -2   -2  -40  -40  -11  -11  -40   33   33  -11  -11
    // -40  -40  -40  -40   44  -22   18   18   29   18  -40  -40  -40  -40  -40
    // -40  -20  -11  -40   29  -40  -40  -40  -40  -40   29  -40  -40  -40  -40
    // -40  -40  -22  -11  -40  -40  -40   29  -11  -40  -22   -7  -40
    REQUIRE_THAT(spectral_alignment.getScoreRow(10),
                 Catch::Matchers::Approx(row10));

    // MaxScore=58 i=12 j=117
    REQUIRE(spectral_alignment.getMaxScore() == 58);
    REQUIRE(spectral_alignment.getMaxPosIterator().index1() == 12);
    REQUIRE(spectral_alignment.getMaxPosIterator().index2() == 117);

    REQUIRE(spectral_alignment.getPrecursorMzDelta() ==
            Catch::Approx(0.0001257147848718887));

    // INFOS:
    // ;AIADGSLLDLLR++++0,00;11;11;[86,09]A[I][-86,09]ADGSLLDLLR_[-0,00];AIADGSLLDLLR;2;0,00;58;0.3224648627994288
    //  INFOS: _hitModifiedSeq=[86,09]A[I][-86,09]ADGSLLDLLR_[-0,00]
    //  _modificationNumber=2
    REQUIRE(spectral_alignment.backTrack().toStdString() ==
            "[86.0926]A[I][-86.0908]ADGSLLDLLR_[-0.00167327]");
  }


  SECTION("..:: align with wrong sequence AIADGSSAMLLDLLR ::..", "[alignwrong]")
  {

    pappso::PeptideSp sampler(
      std::make_shared<pappso::Peptide>("AIADGSSAMLLDLLR"));

    pappso::specglob::PeptideSpectraCsp peptide_spectra =
      std::make_shared<pappso::specglob::PeptideSpectrum>(sampler);

    QualifiedMassSpectrum spectrum_simple =
      readMgf(QString(CMAKE_SOURCE_DIR)
                .append("/tests/data/peaklists/peaklist_15046.mgf"));
    pappso::FilterGreatestY(60).filter(
      *(spectrum_simple.getMassSpectrumSPtr().get()));

    PrecisionPtr precision_ptr =
      pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::specglob::ExperimentalSpectrumCsp experimental_spectrum =
      std::make_shared<pappso::specglob::ExperimentalSpectrum>(spectrum_simple,
                                                               precision_ptr);
    pappso::specglob::SpectralAlignment spectral_alignment(
      pappso::specglob::ScoreValues(), precision_ptr);

    spectral_alignment.align(peptide_spectra, experimental_spectrum);

    // INFOS:
    // ;AIADGSSAMLLDLLR++++-289,11;7;11;[A][I][A][D][G][S][S][A][M][-289,11]LLDLLR_[-0,00];[A]IADGSSAM[-289,11]LLDLLR;1;0,00;27;0.3224648627994288
    REQUIRE(spectral_alignment.backTrack().toStdString() ==
            "[A][I][A][D][G][S][S][A][M][-289.109]LLDLLR_[-0.000870138]");
  }
}
