/**
 * \file tests/specglobspectra.cpp
 * \date 07/11/2023
 * \author Olivier Langella
 * \brief tests for spectra objects
 */


/*
 * SpecGlobTool, Spectra to peptide alignment tool
 * Copyright (C) 2023  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
// ./tests/catch2-only-tests [spectra] -s


#include <QDebug>
#include <QString>
#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>
#include "config.h"
#include "common.h"
#include <pappsomspp/core/massspectrum/massspectrum.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include <pappsomspp/core/processing/filters/filterpass.h>
#include <pappsomspp/core/processing/specglob/experimentalspectrum.h>

TEST_CASE("specglob test experimental spectrum", "[spectra]")
{
  // Set the debugging message formatting pattern.scores to apply in comparisons
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: mzIdentML parser init ::..", "[spectra]")
  {

    /*
     * INFOS: FILTER_TYPE=1 SGXProperties.N_MOST_INTENSE=60
nov. 08, 2023 9:38:39 AM tests.TheoreticalSpectrumTest main
INFOS: 1
1.007825032241	0
87.044518	0
87.100098	0
89.060104	0
101.071022	0
102.055191	0
103.0392	0
110.071404	0
120.08078	0
129.102066	0
130.086136	0
133.085938	0
136.07547	0
141.101654	0
143.117661	0
147.112473	0
157.133316	0
158.091949	0
158.137009	0
159.091446	0
169.097122	0
171.112518	0
173.128479	0
175.118683	0
183.112579	0
184.1287930000001	1
185.128174	2
185.163437	0
186.12265	0
197.127594	0
199.180481	0
201.123016	0
214.118256	2
215.138428	0
229.117401	0
255.16486400000008	1
256.167367	1
271.175751	0
288.202057	0
308.188263	0
309.192413	0
330.169708	0
342.202484	0
370.190194	1
371.193673	1
401.286804	0
402.290344	0
414.2143030000001	1
427.21332600000005	1
428.21692700000006	1
428.269226	0
473.963409	0
514.2462850000001	1
515.2476280000001	1
516.313049	0
517.317139	0
614.31318	1
627.322457	1
628.3310630000001	1
629.397217	0
630.405823	0
643.4151	0
740.411141	1
741.4152310000001	1
742.480652	0
743.481995	0
783.7648710000001	1
829.459054	1
829.511353	0
830.514954	0
843.513977	0
855.437936	1
856.441476	1
886.534607	0
887.538086	0
915.525796	1
927.558572	1
948.535867	1
949.540017	1
969.5262230000001	1
986.552529	1
1001.560913	0
1002.563416	0
1028.610879	1
1042.589852	1
1043.614746	2
1056.605264	1
1058.547799	1
1060.600686	1
1071.60563	1
1072.564843	1
1072.599121	2
1073.599487	0
1074.615701	1
1082.6095970000001	1
1084.599801	1
1086.6157620000001	1
1088.6311580000001	1
1098.6368340000001	1
1099.591271	1
1099.6363310000002	1
1100.5949640000001	1
1110.6158070000001	1
1114.610619	1
1116.626626	1
1121.65281	1
1124.642342	1
1127.642144	1
1128.6262140000001	1
1137.6475	1
1147.656876	1
1154.68908	1
1155.6730890000001	1
1156.657258	1
1168.6681760000001	1
1170.628182	1
1170.6837620000001	1
1238.710438849079	0

*/
    QualifiedMassSpectrum spectrum_simple =
      readMgf(QString(CMAKE_SOURCE_DIR)
                .append("/tests/data/peaklists/peaklist_15046.mgf"));
    // pappso::FilterResampleKeepGreater(60).filter(spectrum_simple);
    // pappso::FilterChargeDeconvolution(
    //  pappso::PrecisionFactory::getDaltonInstance(0.02))
    //  .filter(spectrum_simple);
    //.applyCutOff(150).takeNmostIntense(100).applyDynamicRange(100);
    pappso::FilterGreatestY(60).filter(
      *(spectrum_simple.getMassSpectrumSPtr().get()));


    std::vector<double> expected_mass_list(
      {87.044518,   87.100098,   89.060104,   101.071022,  102.055191,
       103.0392,    110.071404,  120.08078,   129.102066,  130.086136,
       133.085938,  136.07547,   141.101654,  143.117661,  147.112473,
       157.133316,  158.091949,  158.137009,  159.091446,  169.097122,
       171.112518,  173.128479,  175.118683,  183.112579,  185.128174,
       185.163437,  186.12265,   197.127594,  199.180481,  201.123016,
       214.118256,  215.138428,  229.117401,  271.175751,  288.202057,
       308.188263,  309.192413,  330.169708,  342.202484,  401.286804,
       402.290344,  428.269226,  473.963409,  516.313049,  517.317139,
       629.397217,  630.405823,  643.4151,    742.480652,  743.481995,
       829.511353,  830.514954,  843.513977,  886.534607,  887.538086,
       1001.560913, 1002.563416, 1043.614746, 1072.599121, 1073.599487});

    REQUIRE_THAT(spectrum_simple.getMassSpectrumCstSPtr().get()->xValues(),
                 Catch::Matchers::Approx(expected_mass_list).margin(0.00001));

    pappso::specglob::ExperimentalSpectrum experimental_spectrum(
      spectrum_simple, pappso::PrecisionFactory::getDaltonInstance(0.02));


    REQUIRE(experimental_spectrum.getTargetMzSum() ==
            Catch::Approx(1257.72828));
    REQUIRE(experimental_spectrum.getTargetMzSum() - pappso::MASSH2O -
              pappso::MPROTIUM ==
            Catch::Approx(1238.710438849079));

    //
    REQUIRE(experimental_spectrum.getSymetricMz(887.538086) ==
            Catch::Approx(370.190194));
    REQUIRE(experimental_spectrum.getSymetricMz(402.290344) ==
            Catch::Approx(855.437936));
    REQUIRE(experimental_spectrum.getSymetricMz(1168.6681760000001) ==
            Catch::Approx(89.060104));


    std::vector<double> expected_both_mass_list(
      {185.128174, 214.118256, 1043.614746, 1072.599121});

    REQUIRE_THAT(
      experimental_spectrum.getMassList(
        pappso::specglob::ExperimentalSpectrumDataPointType::both),
      Catch::Matchers::Approx(expected_both_mass_list).margin(0.00001));


    std::vector<double> expected_native_mass_list(
      {87.044518,   87.100098,  89.060104,  101.071022, 102.055191, 103.0392,
       110.071404,  120.08078,  129.102066, 130.086136, 133.085938, 136.07547,
       141.101654,  143.117661, 147.112473, 157.133316, 158.091949, 158.137009,
       159.091446,  169.097122, 171.112518, 173.128479, 175.118683, 183.112579,
       185.163437,  186.12265,  197.127594, 199.180481, 201.123016, 215.138428,
       229.117401,  271.175751, 288.202057, 308.188263, 309.192413, 330.169708,
       342.202484,  401.286804, 402.290344, 428.269226, 473.963409, 516.313049,
       517.317139,  629.397217, 630.405823, 643.4151,   742.480652, 743.481995,
       829.511353,  830.514954, 843.513977, 886.534607, 887.538086, 1001.560913,
       1002.563416, 1073.599487});

    REQUIRE_THAT(
      experimental_spectrum.getMassList(
        pappso::specglob::ExperimentalSpectrumDataPointType::native),
      Catch::Matchers::Approx(expected_native_mass_list).margin(0.00001));


    std::vector<double> expected_symetric_mass_list(
      {184.1287930000001,  255.16486400000008, 256.167367,
       370.190194,         371.193673,         414.2143030000001,
       427.21332600000005, 428.21692700000006, 514.2462850000001,
       515.2476280000001,  614.31318,          627.322457,
       628.3310630000001,  740.411141,         741.4152310000001,
       783.7648710000001,  829.459054,         855.437936,
       856.441476,         915.525796,         927.558572,
       948.535867,         949.540017,         969.5262230000001,
       986.552529,         1028.610879,        1042.589852,
       1056.605264,        1058.547799,        1060.600686,
       1071.60563,         1072.564843,        1074.615701,
       1082.6095970000001, 1084.599801,        1086.6157620000001,
       1088.6311580000001, 1098.6368340000001, 1099.591271,
       1099.6363310000002, 1100.5949640000001, 1110.6158070000001,
       1114.610619,        1116.626626,        1121.65281,
       1124.642342,        1127.642144,        1128.6262140000001,
       1137.6475,          1147.656876,        1154.68908,
       1155.6730890000001, 1156.657258,        1168.6681760000001,
       1170.628182,        1170.6837620000001});

    REQUIRE_THAT(
      experimental_spectrum.getMassList(
        pappso::specglob::ExperimentalSpectrumDataPointType::symmetric),
      Catch::Matchers::Approx(expected_symetric_mass_list).margin(0.00001));

    std::vector<double> expected_synthetic_mass_list(
      {1.0078250322, 1238.7098902837});
    REQUIRE_THAT(
      experimental_spectrum.getMassList(
        pappso::specglob::ExperimentalSpectrumDataPointType::synthetic),
      Catch::Matchers::Approx(expected_synthetic_mass_list).margin(0.00001));

    REQUIRE(experimental_spectrum.getPrecursorMass() ==
            Catch::Approx(1255.7137270662));

    REQUIRE(experimental_spectrum.getTargetMzSum() ==
            Catch::Approx(1257.72828));

    pappso::PrecisionPtr precision =
      pappso::PrecisionFactory::getDaltonInstance(0.02);
    pappso::MzRange mz_range(1170.628182 - 1156.657258, precision);

    REQUIRE(experimental_spectrum.size() == 118);
    REQUIRE(experimental_spectrum.reverseFindDiffMz(
              experimental_spectrum.size() - 3, mz_range) !=
            experimental_spectrum.rend());
    REQUIRE(experimental_spectrum
              .reverseFindDiffMz(experimental_spectrum.size() - 3, mz_range)
              ->indice == 113);
    REQUIRE(experimental_spectrum.reverseFindDiffMz(115, mz_range) !=
            experimental_spectrum.rend());
    REQUIRE(experimental_spectrum.reverseFindDiffMz(115, mz_range)->indice ==
            113);

    pappso::MzRange mz_range2(156.101, precision);
    // start_position 15  lookfor= 156.101
    REQUIRE(experimental_spectrum.reverseFindDiffMz(15, mz_range2) ==
            experimental_spectrum.rend());
  }
}
