/**
 * \file tests/matrix.cpp
 * \date 11/11/2023
 * \author Olivier Langella
 * \brief tests on boost matrix
 */


/*
 * SpecGlobTool, Spectra to peptide alignment tool
 * Copyright (C) 2023  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
// ./tests/catch2-only-tests [matrix] -s

// ./tests/catch2-only-tests [benchmark] >>
// ../tests/benchmark/results/benchmark.txt


#include <QDebug>
#include <QString>

#define CATCH_CONFIG_ENABLE_BENCHMARKING
#include <catch2/catch_test_macros.hpp>
#include <catch2/benchmark/catch_benchmark.hpp>
#include "config.h"
#include "common.h"
#include <pappsomspp/core/massspectrum/massspectrum.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include <pappsomspp/core/processing/filters/filterpass.h>
#include <pappsomspp/core/processing/specglob/spectralalignment.h>


TEST_CASE("specglob benchmark", "[benchmark]")
{
  // Set the debugging message formatting pattern.scores to apply in comparisons
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));


  SECTION("..:: align with wrong sequence AIADGSSAMLLDLLR ::..", "[benchmark]")
  {

    pappso::PeptideSp sampler(
      std::make_shared<pappso::Peptide>("AIADGSSAMLLDLLR"));

    pappso::specglob::PeptideSpectraCsp peptide_spectra =
      std::make_shared<pappso::specglob::PeptideSpectrum>(sampler);

    QualifiedMassSpectrum spectrum_simple =
      readMgf(QString(CMAKE_SOURCE_DIR)
                .append("/tests/data/peaklists/peaklist_15046.mgf"));
    pappso::FilterGreatestY(60).filter(
      *(spectrum_simple.getMassSpectrumSPtr().get()));

    PrecisionPtr precision_ptr =
      pappso::PrecisionFactory::getDaltonInstance(0.02);

    pappso::specglob::ExperimentalSpectrumCsp experimental_spectrum =
      std::make_shared<pappso::specglob::ExperimentalSpectrum>(spectrum_simple,
                                                               precision_ptr);
    pappso::specglob::SpectralAlignment spectral_alignment(
      pappso::specglob::ScoreValues(), precision_ptr);


    BENCHMARK("spectral_alignment.align AIADGSSAMLLDLLR vs peaklist_15046")
    {
      spectral_alignment.align(peptide_spectra, experimental_spectrum);
    };
    REQUIRE(spectral_alignment.backTrack().toStdString() ==
            "[A][I][A][D][G][S][S][A][M][-289.109]LLDLLR_[-0.000870138]");
  }
}
