/**
 * \file core/peptidoms.h
 * \date 12/09/2025
 * \author Olivier Langella
 * \brief Running peptidoms
 */


/*
 * PeptidOMS, Spectra to protein alignment tool
 * Copyright (C) 2025  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#pragma once

#include <QJsonObject>
#include <QFileInfo>
#include <pappsomspp/core/processing/uimonitor/uimonitorinterface.h>
#include <pappsomspp/core/msrun/spectrumcollectionhandlerinterface.h>
#include <pappsomspp/core/processing/cbor/cborstreamwriter.h>
#include <pappsomspp/core/amino_acid/aacode.h>
#include <pappsomspp/core/processing/specpeptidoms/scorevalues.h>
#include <pappsomspp/core/fasta/fastahandlerinterface.h>
#include <pappsomspp/core/protein/protein.h>
#include <pappsomspp/core/processing/specpeptidoms/semiglobalalignment.h>
#include "../output/peptidomscboroutput.h"
#include "onescanprocess.h"
/**
 * @todo write docs
 */
class RunningPeptidOms
{
  friend OneScanProcess;

  public:
  /**
   * Default constructor
   */
  RunningPeptidOms(pappso::UiMonitorInterface &monitor,
                   const QJsonObject &json_params);

  /**
   * Destructor
   */
  virtual ~RunningPeptidOms();

  void run(pappso::MsRunReaderSPtr msrun_reader,
           const std::vector<QFileInfo> &fasta_fileinfo_list,
           pappso::cbor::CborStreamWriter *p_cbor_stream_writer);

  private:
  class FastaSeqHandler : public pappso::FastaHandlerInterface
  {
    public:
    FastaSeqHandler(RunningPeptidOms *p_parent)
    {
      mp_parent = p_parent;
    }
    void
    setSequence(const QString &description_in,
                const QString &sequence_in) override
    {
      QString sequence = sequence_in.trimmed().simplified();
      if(!sequence.contains("X"))
        {
          mp_parent->m_proteinList.push_back(
            {description_in, sequence, mp_parent->m_aaCode});
        }
    };

    private:
    RunningPeptidOms *mp_parent;
  };

  class SpectrumCollectionHandler
    : public pappso::SpectrumCollectionHandlerInterface
  {
    public:
    SpectrumCollectionHandler(RunningPeptidOms *p_parent,
                              pappso::UiMonitorInterface &monitor)
      : m_monitor(monitor)
    {
      mp_parent = p_parent;
    }
    virtual ~SpectrumCollectionHandler() {};

    virtual void setQualifiedMassSpectrum(
      const pappso::QualifiedMassSpectrum &spectrum) override;

    virtual bool
    needPeakList() const override
    {
      return true;
    };

    private:
    pappso::UiMonitorInterface &m_monitor;
    RunningPeptidOms *mp_parent;
  };

  void loadProteinList(const QFileInfo &fasta_file_info);

  void processScanResultArray();

  void onOneScanProcessStarted(std::size_t spectrum_index);
  void onOneScanProcessFinished(std::size_t spectrum_index);

  private:
  pappso::UiMonitorInterface &m_monitor;
  QJsonObject m_peptidOmsParameters;
  pappso::PrecisionPtr m_fragmentTolerance;
  double m_minimumMz;
  std::size_t m_nMostIntense;
  bool m_deisotope = true;
  pappso::AaCode m_aaCode;
  pappso::specpeptidoms::ScoreValues m_scoreValues;
  std::vector<pappso::specpeptidoms::SpOMSProtein> m_proteinList;

  PeptidOmsCborOutput *mpa_peptidOmsCborOutput = nullptr;

  std::vector<OneScanProcess *> m_scanResultArray;


  std::size_t m_maxInterpretationsPerSpectrum = 1;


  std::vector<std::size_t> m_runningSpectrumStack;

  std::size_t m_countScanFinished = 0;

  QMutex m_mutex;
};
