/**
 * \file tests/specglobpeptidespectra.cpp
 * \date 08/11/2023
 * \author Olivier Langella
 * \brief tests for peptide spectra objects
 */


/*
 * SpecGlobTool, Spectra to peptide alignment tool
 * Copyright (C) 2023  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
// ./tests/catch2-only-tests [peptide] -s
/*
langella@piccolo:~/developpement/git/specglobtool/cbuild$ java -jar
bin/TestSpecGlob.jar nov. 06, 2023 4:06:20 PM tests.TheoreticalSpectrumTest main
INFOS: SAMPLER
1.007276466879	10000.0
88.03930487200401	10000.0
159.076418657569	10000.0
290.116903572098	10000.0
387.169667422145	10000.0
500.253731401156	10000.0
629.2963244903231	10000.0
785.3974355159751	10000.0
*/
//{ 88.039304872, 159.0764186576, 290.1169037462, 387.1696675962, 500.
//  2537315753, 629.2963246644 }


#include <QDebug>
#include <QString>
#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>
#include "config.h"
#include <pappsomspp/core/peptide/peptide.h>
#include <pappsomspp/core/processing/specglob/peptidespectrum.h>


TEST_CASE("specglog test peptide spectrum", "[peptide]")
{
  // Set the debugging message formatting pattern.scores to apply in comparisons
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: mzIdentML parser init ::..", "[peptide]")
  {
    pappso::PeptideSp sampler(std::make_shared<pappso::Peptide>("SAMPLER"));

    pappso::specglob::PeptideSpectrum peptide_spectra(sampler);
    std::vector<double> expected_mass_list({1.007276466879,
                                            88.03930487200401,
                                            159.076418657569,
                                            290.116903572098,
                                            387.169667422145,
                                            500.253731401156,
                                            629.2963244903231,
                                            785.3974355159751});

    REQUIRE_THAT(peptide_spectra.getMassList(),
                 Catch::Matchers::Approx(expected_mass_list).margin(0.00001));

    REQUIRE(peptide_spectra[1].diff_mz ==
            Catch::Approx(88.03930487200401 - 1.007276466879));
    REQUIRE(peptide_spectra[2].diff_mz ==
            Catch::Approx(159.076418657569 - 88.03930487200401));
    REQUIRE(peptide_spectra[7].diff_mz ==
            Catch::Approx(785.397435516 - 629.2963244903231));
  }
}
