/**
 * \file output/psmcboroutput.h
 * \date 27/06/2026
 * \author Olivier Langella
 * \brief write psm in a simple cbor file
 */

/*******************************************************************************
 * Copyright (c) 2026 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of SpecOMS.
 *
 *     SpecOMS is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     SpecOMS is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with SpecOMS.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "psmcboroutput.h"
#include <QThreadPool>
#include <QCborStreamReader>
#include <QCborValue>
#include <QCborMap>
#include <pappsomspp/core/utils.h>
#include <pappsomspp/core/pappsoexception.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include <pappsomspp/core/processing/filters/filterpass.h>
#include <pappsomspp/core/processing/specglob/peptidespectrum.h>
#include <pappsomspp/core/processing/specglob/spectralalignment.h>
#include <pappsomspp/core/processing/filters/filterchargedeconvolution.h>
#include "../core/fastasourcelist.h"

PsmCborOutput::PsmCborOutput(const QString &outputFileStr)
  : mpa_file(outputFileStr)
{
  mpa_file.open(QIODeviceBase::WriteOnly);
  mpa_writer = new pappso::cbor::CborStreamWriter(&mpa_file);
  mpa_writer->startMap();
  mpa_writer->writeInformations(SOFTWARE_NAME, VERSION, "psm", "specoms");


  mpa_writer->append("parameter_map");
  mpa_writer->startMap();
  mpa_writer->append("specoms");
  // mpa_writer->append("specoms");
  //  mpa_writer->append("test");
  QCborMap specoms_map;
  QJsonObject json_map;
  DeepProtParams::getInstance().toJsonObject(json_map);
  // qWarning() << json_map.keys();
  //  QJsonValue json_value(json_map);
  specoms_map = QCborMap::fromJsonObject(json_map);
  specoms_map.toCborValue().toCbor(*mpa_writer);
  mpa_writer->endMap();

  // QByteArray byteArray
  mpa_buffer = new QBuffer();
  mpa_buffer->open(QIODevice::ReadWrite);
  mpa_bufferWriter = new pappso::cbor::CborStreamWriter(mpa_buffer);
}

PsmCborOutput::~PsmCborOutput()
{
  close();
}

void
PsmCborOutput::close()
{

  mpa_bufferWriter->endArray(); // end scan_list
  mpa_bufferWriter->endMap();   // end sample
  mpa_bufferWriter->endArray(); // end sample_list

  mpa_writer->append("protein_map");
  m_psmProteinMap.writeMap(*mpa_writer);

  // copy buffer

  mpa_writer->append("sample_list");

  // mpa_writer->startArray(); // sample_list
  // mpa_writer->endArray();   // end sample_list


  mpa_buffer->seek(0);
  // qWarning() << "size=" << mpa_buffer->data().size();

  mpa_file.write(mpa_buffer->data());
  /*
  QCborStreamReader cbor_reader(mpa_buffer->data());
  QCborValue map_tag;
  map_tag.fromCbor(cbor_reader);

  map_tag.toCbor(*mpa_writer);
  // map_tag.fromCbor(cbor_reader);

  qWarning() << map_tag.toString();
  //  map_tag.toCbor(*mpa_writer);
*/

  mpa_writer->endMap();
  delete mpa_writer;
  mpa_file.close();
}


void
PsmCborOutput::setDeepProtParams(const DeepProtParams &deepprot_params)
{
  mp_deepProtParams = &deepprot_params;
}

void
PsmCborOutput::setMsRunReaderSPtr(pappso::MsRunReaderSPtr msrun_reader)
{
  msp_msrunReader = msrun_reader;
}

void
PsmCborOutput::setPeptideDatabaseSPtr(PeptideDatabaseSPtr &peptide_database)
{
  msp_peptideDatabase = peptide_database;

  mpa_writer->append("target_fasta_files");
  mpa_writer->startArray();
  const std::vector<FastaFile> fasta_file_list =
    msp_peptideDatabase.get()->getFastaSourceList().get()->getFastaFileList();
  for(auto &fasta_file : fasta_file_list)
    {
      mpa_writer->append(fasta_file.fileName());
    }
  mpa_writer->endArray();


  const std::vector<FastaFile> fasta_file_decoy_list =
    msp_peptideDatabase.get()
      ->getFastaSourceList()
      .get()
      ->getDecoyFastaFileList();
  if(fasta_file_decoy_list.size() > 0)
    {
      mpa_writer->startArray();
      for(const FastaFile &fasta_file : fasta_file_decoy_list)
        {
          mpa_writer->append(fasta_file.fileName());
        }
      mpa_writer->endArray();
    }
}

void
PsmCborOutput::writeSampleHeader(
  pappso::cbor::CborStreamWriter *p_writer,
  const pappso::QualifiedMassSpectrum &experimental_spectrum_int)
{
  // p_writer->append("sample_list");
  /*

  "sample_list": [
  {
  "name": "tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1",
  "identification_file": {
  "name": "/home/langella/data1/tandem/
  tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1.xml",
  },
  "peaklist_file": {
  "name": "tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1.mzml"
  },
  "scan_list": [
  {
  */
  p_writer->startArray();

  p_writer->startMap();
  p_writer->append("name");
  p_writer->append(experimental_spectrum_int.getMassSpectrumId()
                     .getMsRunIdCstSPtr()
                     .get()
                     ->getSampleName());
  p_writer->append("peaklist_file");
  p_writer->startMap();
  p_writer->append("name");
  p_writer->append(experimental_spectrum_int.getMassSpectrumId()
                     .getMsRunIdCstSPtr()
                     .get()
                     ->getFileName());
  p_writer->endMap();

  p_writer->append("scan_list");
  p_writer->startArray();
  m_headerOk = true;
}
void
PsmCborOutput::writePeptideCandidateList(
  const pappso::QualifiedMassSpectrum &experimental_spectrum_int,
  const std::vector<OutputPeptideCandidateStruct> &peptide_candidate_list,
  pappso::DeepProtMatchType match_type)
{

  QMutexLocker locker(&m_mutex);

  pappso::cbor::CborStreamWriter *p_writer = mpa_bufferWriter;

  if(!m_headerOk)
    {
      writeSampleHeader(p_writer, experimental_spectrum_int);
    }
  // std::cout << experimental_spectrum_int.title.toStdString() << endl;
  // std::cout.flush();
  // return;
  if(peptide_candidate_list.size() == 0)
    {

      qDebug();
    }
  else
    {
      // std::vector<PeptideCandidate> sorted = m_peptideCandidates;
      // qDebug() << __FILE__ << " " << __FUNCTION__ << " " << __FUNCTION__ << "
      // ";
      qDebug();
      p_writer->startMap(); // start scan
      /*

                  mpa_calcWriter->writeLine();
                  mpa_calcWriter->writeCell(experimental_spectrum_int.title);
                  mpa_calcWriter->writeCell(experimental_spectrum_int.spectrumIndex);
                  mpa_calcWriter->writeCell(experimental_spectrum_int.charge);
                  return;
        */
      if(experimental_spectrum_int.getPrecursorCharge() == 0)
        {
          throw pappso::PappsoException(
            QObject::tr(
              "ERROR experimental_spectrum_int.charge == 0 in :\n%1 %2 %3")
              .arg(__FILE__)
              .arg(__FUNCTION__)
              .arg(__LINE__));
        }

      // id
      p_writer->append("id");
      p_writer->startMap();
      p_writer->append("index");
      p_writer->append((quint64)experimental_spectrum_int.getMassSpectrumId()
                         .getSpectrumIndex());
      p_writer->append("native_id");
      p_writer->append(
        experimental_spectrum_int.getMassSpectrumId().getNativeId());
      p_writer->endMap();

      // precursor
      p_writer->append("precursor");
      p_writer->startMap();
      p_writer->append("z");
      p_writer->append(experimental_spectrum_int.getPrecursorCharge());
      p_writer->append("mz");
      p_writer->append(experimental_spectrum_int.getPrecursorMz());
      p_writer->append("mh");
      p_writer->append(experimental_spectrum_int.getPrecursorMass());
      p_writer->append("intensity");
      p_writer->append(experimental_spectrum_int.getPrecursorIntensity());
      p_writer->endMap();

      // ms2
      p_writer->append("ms2");
      p_writer->startMap();
      p_writer->append("rt");
      p_writer->append(experimental_spectrum_int.getRtInSeconds());
      msp_realSpectrum = msp_msrunReader.get()->massSpectrumSPtr(
        experimental_spectrum_int.getMassSpectrumId().getSpectrumIndex());

      p_writer->append("mz");
      p_writer->writeArray(msp_realSpectrum.get()->xValues());

      p_writer->append("intensity");
      p_writer->writeArray(msp_realSpectrum.get()->yValues());
      p_writer->endMap();

      // psm_list

      p_writer->append("psm_list");
      p_writer->startArray();

      if(!m_isOutputAllPsmCandidates)
        {
          const OutputPeptideCandidateStruct &peptide_candidate =
            peptide_candidate_list[getBestPeptideCandidateIndex(
              peptide_candidate_list)];

          writePeptideCandidate(p_writer,
                                experimental_spectrum_int,
                                peptide_candidate_list.size(),
                                peptide_candidate,
                                match_type);
        }
      else
        {
          for(auto &&peptide_candidate : peptide_candidate_list)
            {
              writePeptideCandidate(p_writer,
                                    experimental_spectrum_int,
                                    peptide_candidate_list.size(),
                                    peptide_candidate,
                                    match_type);
            }
        }

      p_writer->endArray(); // end psm_list

      p_writer->endMap(); // end scan
    }
}

void
PsmCborOutput::writePeptideCandidate(
  pappso::cbor::CborStreamWriter *p_writer,
  const pappso::QualifiedMassSpectrum &experimental_spectrum_int,
  std::size_t candidate_size,
  const OutputPeptideCandidateStruct &peptide_candidate,
  pappso::DeepProtMatchType match_type)
{
  p_writer->startMap(); // psm

  p_writer->append("proforma");
  p_writer->append(peptide_candidate.originalPeptideString.get()->toProForma());


  p_writer->append("protein_list");
  p_writer->startArray();
  QStringList str_protein_names;
  for(auto protein_index : msp_peptideDatabase->getProteinIndexList(
        peptide_candidate.protoPeptideNum))
    {

      pappso::cbor::psm::PsmProtein psm_protein;
      psm_protein.protein_sp =
        std::make_shared<pappso::Protein>(msp_peptideDatabase.get()
                                            ->getFastaSourceList()
                                            .get()
                                            ->getProteinByIndex(protein_index));
      psm_protein.isTarget =
        msp_peptideDatabase->isProteinTarget(protein_index);

      m_psmProteinMap.insert(psm_protein);

      p_writer->startMap(); // protein
      p_writer->append("accession");
      p_writer->append(psm_protein.protein_sp.get()->getAccession());


      // mpa_bufferWriter->append("target");
      // mpa_bufferWriter->append(msp_peptideDatabase->isProteinTarget(protein_index));

      // start/end positions
      QString protein_sequence =
        QString(psm_protein.protein_sp.get()->getSequence()).replace("L", "I");
      QString peptide_sequence =
        peptide_candidate.originalPeptideString.get()->getSequenceLi();
      int position = protein_sequence.indexOf(peptide_sequence);

      std::vector<int> positions;
      while(position >= 0)
        {
          positions.push_back(position);
          position = protein_sequence.indexOf(peptide_sequence, position + 1);
        }

      p_writer->append("positions");
      p_writer->writeArray(positions);

      p_writer->endMap(); // protein
    }
  p_writer->endArray(); // protein_list


  p_writer->append("eval");
  p_writer->startMap(); // eval
  p_writer->append("specoms");
  p_writer->startMap(); // specoms
  p_writer->append("candidate_size");
  p_writer->append((quint64)candidate_size);
  p_writer->append("match_type");
  p_writer->append((int)match_type);
  p_writer->append("status");
  p_writer->append((int)peptide_candidate.status);
  p_writer->append("massDelta");
  p_writer->append(peptide_candidate.massDelta);
  p_writer->append("count");
  p_writer->append((int)peptide_candidate.originalCount);
  p_writer->endMap(); // specoms


  p_writer->append("specfit");
  p_writer->startMap(); // specfit
  p_writer->append("count");
  p_writer->append((int)peptide_candidate.fittedCount);
  p_writer->append("proforma");
  p_writer->append(getRefinedPeptideSp(peptide_candidate.peptideModel,
                                       peptide_candidate.massDelta,
                                       peptide_candidate.deltaPositions)
                     .get()
                     ->toProForma());

  if(peptide_candidate.peptideModel.get() != nullptr)
    {

      p_writer->append("deltaPositions");
      p_writer->writeArray(peptide_candidate.deltaPositions);
    }


  p_writer->endMap(); // specfit

  if(m_isSpecGlob)
    {
      writeSpecGlobEval(p_writer, experimental_spectrum_int, peptide_candidate);
    }
  p_writer->endMap(); // eval


  p_writer->endMap(); // psm
}


void
PsmCborOutput::writeSpecGlobEval(
  pappso::cbor::CborStreamWriter *p_writer,
  const pappso::QualifiedMassSpectrum &experimental_spectrum_int,
  const OutputPeptideCandidateStruct &peptide_candidate)
{
  pappso::specglob::PeptideSpectraCsp peptide_spectra =
    std::make_shared<pappso::specglob::PeptideSpectrum>(
      peptide_candidate.originalPeptideString);

  pappso::QualifiedMassSpectrum spectrum_simple = experimental_spectrum_int;
  spectrum_simple.setMassSpectrumSPtr(msp_realSpectrum);

  pappso::PrecisionPtr precision_ptr =
    pappso::PrecisionFactory::getDaltonInstance(0.02);
  pappso::FilterChargeDeconvolution(precision_ptr)
    .filter(*(spectrum_simple.getMassSpectrumSPtr().get()));
  pappso::FilterResampleKeepGreater(150).filter(
    *(spectrum_simple.getMassSpectrumSPtr().get()));
  pappso::FilterGreatestY(120).filter(
    *(spectrum_simple.getMassSpectrumSPtr().get()));


  pappso::specglob::ExperimentalSpectrumCsp experimental_spectrum =
    std::make_shared<pappso::specglob::ExperimentalSpectrum>(spectrum_simple,
                                                             precision_ptr);
  pappso::specglob::SpectralAlignment spectral_alignment(
    pappso::specglob::ScoreValues(), precision_ptr);

  spectral_alignment.align(peptide_spectra, experimental_spectrum);

  p_writer->append("specglob");
  p_writer->startMap(); // specglob

  /*
  // debug
  mpa_bufferWriter->append("spec_peptide");
  mpa_bufferWriter->append(peptide_spectra.get()->getPeptideSp().get()->toProForma());
  mpa_bufferWriter->append("prec_mz");
  mpa_bufferWriter->append(spectrum_simple.getPrecursorMz());
  mpa_bufferWriter->append("mzl");
  mpa_bufferWriter->append(
    (int)spectrum_simple.getMassSpectrumCstSPtr().get()->xValues().size());
  mpa_bufferWriter->append("exp_spec");
  mpa_bufferWriter->append(experimental_spectrum.get()->toString());
  // debug
*/
  p_writer->append("max_score");
  p_writer->append(spectral_alignment.getMaxScore());
  if(spectral_alignment.getMaxScore() > 0)
    {
      pappso::specglob::PeptideModel peptide_model =
        spectral_alignment.buildPeptideModel();
      peptide_model.eliminateComplementaryDelta(precision_ptr);
      peptide_model.matchExperimentalPeaks(precision_ptr);

      p_writer->append("count");
      p_writer->append((quint64)peptide_model.getCountSharedPeaks());

      p_writer->append("proforma");
      p_writer->append(peptide_model.toProForma());
    }

  p_writer->endMap(); // specglob
}
