/**
 * \file pappsomspp/core/processing/cbor/mzcbor/cvparam.cpp
 * \date 23/11/2025
 * \author Olivier Langella
 * \brief PSI cvParam object for mzML/mzCBOR
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "cvparam.h"
#include "pappsomspp/core/pappsoexception.h"


void
pappso::cbor::mzcbor::CvParam::fromCbor(CborStreamReader &reader)
{
  qDebug();
  reader.enterContainer();
  QString attribute_cvparam;
  while(reader.hasNext())
    {
      reader.decodeString(attribute_cvparam);
      qDebug() << attribute_cvparam;
      if(attribute_cvparam == "cvRef")
        {
          reader.decodeString(cvRef);
        }
      else if(attribute_cvparam == "accession")
        {
          reader.decodeString(accession);
        }
      else if(attribute_cvparam == "name")
        {
          reader.decodeString(name);
        }
      else if(attribute_cvparam == "value")
        {
          cborType = reader.type();
          if(reader.isDouble())
            {
              valueDouble = reader.toDouble();
              reader.next();
            }
          else if(reader.isUnsignedInteger())
            {
              valueInt = reader.toUnsignedInteger();
              reader.next();
            }
          else if(reader.isInteger())
            {
              valueInt = reader.toInteger();
              reader.next();
            }
          else if(reader.type() == QCborStreamReader::Type::String)
            {
              if(reader.decodeString(attribute_cvparam))
                {
                  valueStr = attribute_cvparam;
                }
              else
                {
                  throw pappso::PappsoException(
                    QObject::tr("cvParam value string failed for accession %1").arg(accession));
                }
            }
          else
            {
              throw pappso::PappsoException(
                QObject::tr("cvParam value type not known for accession %1").arg(accession));
            }
        }
      else if(attribute_cvparam == "unitAccession")
        {
          reader.decodeString(unitAccession);
        }

      else if(attribute_cvparam == "unitCvRef")
        {
          reader.decodeString(unitCvRef);
        }
      else if(attribute_cvparam == "unitName")
        {
          reader.decodeString(unitName);
        }
      else
        {
          reader.next();
        }
    }

  reader.leaveContainer();
}


std::map<QString, pappso::cbor::mzcbor::CvParam>
pappso::cbor::mzcbor::CvParam::getCvParamsMapFromCbor(CborStreamReader &reader)
{
  qDebug();
  std::map<QString, CvParam> accession_values;
  reader.enterContainer(); // start array

  while(reader.hasNext())
    {

      CvParam cv_param;
      cv_param.fromCbor(reader);
      accession_values.insert({cv_param.accession, cv_param});
    }

  reader.leaveContainer(); // end array
  return accession_values;
}

void
pappso::cbor::mzcbor::CvParam::setValue(const QString &value_str)
{
  valueInt    = 0;
  valueDouble = 0;
  valueStr    = value_str;
}

void
pappso::cbor::mzcbor::CvParam::toMzml(QXmlStreamWriter &writer)
{
  //          <cvParam cvRef="MS" accession="MS:1000514" value="" name="m/z array"
  writer.writeStartElement("cvParam");

  writer.writeAttribute("cvRef", cvRef);
  writer.writeAttribute("accession", accession);


  if(cborType == QCborStreamReader::Type::Double)
    {
      valueStr = QString::number(valueDouble, 'g', 15);
    }
  else if(cborType == QCborStreamReader::Type::UnsignedInteger)
    {
      valueStr = QString("%1").arg(valueInt);
    }
  else if(cborType == QCborStreamReader::Type::NegativeInteger)
    {
      valueStr = QString("%1").arg(valueInt);
    }
  else if(cborType == QCborStreamReader::Type::String)
    {
    }
  writer.writeAttribute("value", valueStr);

  writer.writeAttribute("name", name);
  //          unitAccession="MS:1000040" unitName="m/z" unitCvRef="MS" />
  if(!unitAccession.isEmpty())
    {
      writer.writeAttribute("unitAccession", unitAccession);
      writer.writeAttribute("unitName", unitName);
      writer.writeAttribute("unitCvRef", unitCvRef);
    }

  writer.writeEndElement(); // cvParam
}
