/**
 * \file mzml2mzcborcli.cpp
 * \date 21/11/2025
 * \author Olivier Langella
 * \brief CLI tool to convert mzML to mzCBOR
 */


/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "mzml2mzcborcli.h"
#include <QIODevice>
#include <QCommandLineParser>
#include "pappsomspp/core/processing/uimonitor/uimonitortext.h"
#include "pappsomspp/core/processing/cbor/mzcbor/mzmlconvert.h"
#include "pappsomspp/core/pappsoexception.h"
#include <QDateTime>
#include <QFile>
#include <QFileInfo>
#include <QTimer>

Mzml2MzcborCli::Mzml2MzcborCli(QObject *parent) : QObject(parent)
{
  // get the instance of the main application
  app = QCoreApplication::instance();
  // setup everything here
  // create any global objects
  // setup debug and warning mode
}

Mzml2MzcborCli::~Mzml2MzcborCli()
{
}

// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void
Mzml2MzcborCli::run()
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  QTextStream errorStream(stderr, QIODevice::WriteOnly);
  QTextStream outputStream(stderr, QIODevice::WriteOnly);

  try
    {
      qDebug();
      QCommandLineParser parser;

      // throw pappso::PappsoException("test");
      // throw pappso::PappsoException("test");
      parser.setApplicationDescription(QString("mzml2mzcbor")
                                         .append(" ")
                                         .append(PAPPSOMSPP_VERSION)
                                         .append(" mzML to mzCBOR converter"));

      parser.addHelpOption();
      parser.addVersionOption();

      QCommandLineOption mzmlInputOption(
        QStringList() << "i" << "input",
        QCoreApplication::translate("mzML file", "PSI mzML file <mzML> or use '-' to read data from stdin."),
        QCoreApplication::translate("mzML", "mzML"));

      QCommandLineOption mzcborOutputOption(
        QStringList() << "o" << "output",
        QCoreApplication::translate("mzCBOR", "path to the mzCBOR output file <mzcbor>."),
        QCoreApplication::translate("mzcbor", "mzcbor"));

      parser.addOption(mzmlInputOption);
      parser.addOption(mzcborOutputOption);

      qDebug();

      // Process the actual command line arguments given by the user
      parser.process(*app);

      // QCoreApplication * app(this);
      // Add your main code here
      // qDebug();
      const QDateTime dt_begin = QDateTime::currentDateTime();
      const QStringList args   = parser.positionalArguments();


      qDebug();
      pappso::UiMonitorText monitor(outputStream);


      // PsmProcess cbor_psm_process(json_parameters);

      QFile input_mzml_file;
      if(!parser.isSet(mzmlInputOption))
        {
          if(parser.positionalArguments().contains("-"))
            {
              input_mzml_file.open(stdin, QIODevice::ReadOnly);
            }
          else
            {
              throw pappso::PappsoException(QObject::tr(
                "input file is not set. Please provide --input or use - to read from <stdin>"));
            }
        }
      else
        {
          input_mzml_file.setFileName(parser.value(mzmlInputOption));
          if(!input_mzml_file.open(QIODevice::ReadOnly))
            {
              throw pappso::PappsoException(QObject::tr("Unable to open mzML file '%1' :\n%2\n")
                                              .arg(QFileInfo(input_mzml_file).absoluteFilePath())
                                              .arg(input_mzml_file.errorString()));
            }
        }

      QFile mzcbor_file;
      pappso::cbor::CborStreamWriter *p_cborWriter = nullptr;
      if(parser.isSet(mzcborOutputOption))
        {
          QString mzcbor_filename = parser.value(mzcborOutputOption);
          if(!mzcbor_filename.isEmpty())
            {
              mzcbor_file.setFileName(mzcbor_filename);
              mzcbor_file.open(QIODevice::WriteOnly);
              p_cborWriter = new pappso::cbor::CborStreamWriter(&mzcbor_file);
            }
        }
      else
        {
          mzcbor_file.open(stdout, QIODevice::WriteOnly);
          p_cborWriter = new pappso::cbor::CborStreamWriter(&mzcbor_file);
        }


      if(p_cborWriter != nullptr)
        {
          pappso::cbor::mzcbor::MzmlConvert mzml_convert(&monitor, p_cborWriter);

          mzml_convert.read(&input_mzml_file);
          input_mzml_file.close();
          mzcbor_file.close();
          if(!mzml_convert.errorString().isEmpty())
            {
              throw pappso::PappsoException(QObject::tr("ERROR converting mzML to mzCBOR :\n%1")
                                              .arg(mzml_convert.errorString()));
            }
        }


      qDebug();
    }
  catch(pappso::PappsoException &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.qwhat());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }

  catch(std::exception &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.what());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }


  // you must call quit when complete or the program will stay in the
  // messaging loop
  quit();
}

// call this routine to quit the application
void
Mzml2MzcborCli::quit()
{
  // you can do some cleanup here
  // then do emit finished to signal CoreApplication to quit
  emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void
Mzml2MzcborCli::aboutToQuitApp()
{
  // stop threads
  // sleep(1);   // wait for threads to stop.
  // delete any objects
}


int
main(int argc, char **argv)
{
  // QTextStream consoleErr(stderr);
  // QTextStream consoleOut(stdout, QIODevice::WriteOnly);
  // ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
  // ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
  qDebug();
  QCoreApplication app(argc, argv);
  qDebug();
  QCoreApplication::setApplicationName("mzml2mzcbor");
  QCoreApplication::setApplicationVersion(PAPPSOMSPP_VERSION);
  QLocale::setDefault(QLocale::system());

  // create the main class
  Mzml2MzcborCli myMain;
  // connect up the signals
  QObject::connect(&myMain, SIGNAL(finished()), &app, SLOT(quit()));
  QObject::connect(&app, SIGNAL(aboutToQuit()), &myMain, SLOT(aboutToQuitApp()));
  qDebug();


  // This code will start the messaging engine in QT and in
  // 10ms it will start the execution in the MainClass.run routine;
  QTimer::singleShot(10, &myMain, SLOT(run()));
  return app.exec();
}
