/**
 * \file src/cli/export/cborstreamreaderalignment.cpp
 * \date 18/07/2025
 * \author Olivier Langella
 * \brief read cbor stream to export alignment data
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of MassChroQ.
 *
 *     MassChroQ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     MassChroQ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with MassChroQ.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "cborstreamreaderalignment.h"
#include <odsstream/odsexception.h>
#include <pappsomspp/core/pappsoexception.h>
#include <QCborArray>

CborStreamReaderAlignment::CborStreamReaderAlignment(
  const QString &output_directory)
{

  qDebug();
  try
    {
      mpa_calcWriterInterface = new TsvDirectoryWriter(QDir(output_directory));
      mpa_calcWriterInterface->setFlushLines(true);
    }
  catch(OdsException &error)
    {
      throw pappso::PappsoException(
        QObject::tr("Error writing TSV ouput in directory %1 :\n%2")
          .arg(output_directory)
          .arg(error.qwhat()));
    }

  qDebug();
}


CborStreamReaderAlignment::~CborStreamReaderAlignment()
{
}

void
CborStreamReaderAlignment::readAlignmentData(
  pappso::UiMonitorInterface &monitor)
{
  QCborMap cbor_alignment_data;
  mpa_cborReader->readCborMap(cbor_alignment_data);

  if(cbor_alignment_data.contains(QString("group_id")))
    {
      m_groupId = cbor_alignment_data.value("group_id").toString();
    }
  else
    {
      throw pappso::PappsoException(
        QObject::tr("group_id not found in alignment_data"));
    }
  if(cbor_alignment_data.contains(QString("alignment")))
    {
    }
  else
    {
      throw pappso::PappsoException(
        QObject::tr("alignment not found in alignment_data %1").arg(m_groupId));
    }

  QCborMap cbor_corrections;
  if(cbor_alignment_data.value(QString("alignment"))
       .toMap()
       .contains(QString("corrections")))
    {
      cbor_corrections = cbor_alignment_data.value(QString("alignment"))
                           .toMap()
                           .value("corrections")
                           .toMap();

      for(auto it : cbor_corrections)
        {
          writeOneAlignment(it.first.toString(), it.second.toMap());
        }
    }
  else
    {
      throw pappso::PappsoException(
        QObject::tr("corrections not found in alignment %1").arg(m_groupId));
    }
  qDebug();
}

void
CborStreamReaderAlignment::writeOneAlignment(const QString &msrun_id,
                                             QCborMap cbor_corrections)
{
  mpa_calcWriterInterface->writeSheet(
    QString("alignment_data_%1-%2").arg(m_groupId).arg(msrun_id));
  qDebug();
  mpa_calcWriterInterface->writeCell("ms2_delta_rt_x");
  mpa_calcWriterInterface->writeCell("ms2_delta_rt_y");
  // mpa_calcWriterInterface->writeCell("original");
  // mpa_calcWriterInterface->writeCell("aligned");
  mpa_calcWriterInterface->writeCell("ms2_median");
  mpa_calcWriterInterface->writeCell("ms2_mean");
  mpa_calcWriterInterface->writeLine();

  if(cbor_corrections.contains(QString("ms2_delta_rt")) &&
     cbor_corrections.contains(QString("ms2_median")) &&
     cbor_corrections.contains(QString("ms2_mean")))
    {
      QCborArray ms2_delta_rt_x =
        cbor_corrections.value(QString("ms2_delta_rt"))
          .toMap()
          .value(QString("x"))
          .toArray();
      QCborArray ms2_delta_rt_y =
        cbor_corrections.value(QString("ms2_delta_rt"))
          .toMap()
          .value(QString("y"))
          .toArray();
      QCborArray original =
        cbor_corrections.value(QString("original")).toArray();
      QCborArray aligned = cbor_corrections.value(QString("aligned")).toArray();
      QCborArray ms2_median =
        cbor_corrections.value(QString("ms2_median")).toArray();
      QCborArray ms2_mean =
        cbor_corrections.value(QString("ms2_mean")).toArray();

      qDebug() << ms2_delta_rt_x.size() << " " << original.size() << " "
               << aligned.size() << " " << ms2_median.size() << " "
               << ms2_mean.size();
      qsizetype i = 0;
      for(auto it : ms2_delta_rt_x)
        {
          mpa_calcWriterInterface->writeCell(ms2_delta_rt_x.at(i).toDouble());
          mpa_calcWriterInterface->writeCell(ms2_delta_rt_y.at(i).toDouble());
          // mpa_calcWriterInterface->writeCell("original");
          // mpa_calcWriterInterface->writeCell("aligned");
          mpa_calcWriterInterface->writeCell(ms2_median.at(i).toDouble());
          mpa_calcWriterInterface->writeCell(ms2_mean.at(i).toDouble());
          mpa_calcWriterInterface->writeLine();
          i++;
        }
    }
}

void
CborStreamReaderAlignment::readQuantificationData(
  pappso::UiMonitorInterface &monitor)
{
  mpa_cborReader->next();
}
