/**
 * \file sagetocborcli.h
 * \date 22/10/2025
 * \author Olivier Langella
 * \brief CLI tool for PeptidOMS project
 */


/*
 * PeptidOMS, Spectra to protein alignment tool
 * Copyright (C) 2025  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/*
./src/cpp/bin/sage2cbor -i /gorgone/pappso/jouy/users/Lydie/2025/2025_TTP/20251006_26_Antigny_frac_TTP2/mzid_sage_all_rattus/results.json -o /tmp/test.cbor
*/

#include "sagetocborcli.h"
#include <sys/stat.h>
#include <QTimer>
#include <QIODevice>
#include <QFile>
#include <QFileInfo>
#include <QCommandLineParser>
#include <pappsomspp/core/pappsoexception.h>
#include <pappsomspp/core/processing/uimonitor/uimonitortext.h>
#include <pappsomspp/core/processing/cbor/psm/sage/sagefilereader.h>
#include "../core/gp_lib_config.h"


  SageToCborCli::SageToCborCli(QObject *parent)
  : QObject(parent)
{
  // get the instance of the main application
  app = QCoreApplication::instance();
  // setup everything here
  // create any global objects
  // setup debug and warning mode
}


SageToCborCli::~SageToCborCli()
{
}

// call this routine to quit the application
void
SageToCborCli::quit()
{
  // you can do some cleanup here
  // then do emit finished to signal CoreApplication to quit
  emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void
SageToCborCli::aboutToQuitApp()
{
  // stop threads
  // sleep(1);   // wait for threads to stop.
  // delete any objects
}


// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void
SageToCborCli::run()
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  //./src/peptidoms
  // -m
  // /gorgone/pappso/data_extraction_pappso/mzML/20120906_balliau_extract_1_A01_urnb-1.mzML
  // -f /gorgone/pappso/data_extraction_pappso/database/Genome_Z_mays_5a.fasta
  // -p params.json
  // -c 10


  QTextStream errorStream(stderr, QIODevice::WriteOnly);
  QTextStream outputStream(stderr, QIODevice::WriteOnly);

  try
    {
      qDebug();
      QCommandLineParser parser;
      pappso::UiMonitorText monitor(errorStream);

      // throw pappso::PappsoException("test");
      parser.setApplicationDescription(
        QString("%1 ")
          .arg(QCoreApplication::applicationName())
          .append("sage2cbor")
          .append(" (")
          .append(GP_VERSION)
          .append(")")
          .append(" -- convert Sage json result file to CBOR PSM"));

      parser.addHelpOption();
      parser.addVersionOption();

      QCommandLineOption sageJsonInputOption(
        QStringList() << "i"
                      << "input",
        QCoreApplication::translate(
          "Sage JSON result file", "path to the Sage JSON result file <sage>."),
        QCoreApplication::translate("Sage", "Sage"));

      QCommandLineOption cborOutputOption(
        QStringList() << "o",
        QCoreApplication::translate(
          "PSM cbor", "path to the PSM cbor output file <output>."),
        QCoreApplication::translate("output", "output"));


      parser.addOption(sageJsonInputOption);
      parser.addOption(cborOutputOption);
      qDebug();

      // Process the actual command line arguments given by the user
      parser.process(*app);

      // QCoreApplication * app(this);
      // Add your main code here
      // qDebug();
      const QStringList args = parser.positionalArguments();


      qDebug();


      qDebug();
      QFile output_cbor_file;
      if(!parser.isSet(cborOutputOption))
        {
          output_cbor_file.open(stdout, QIODevice::WriteOnly);
        }
      else
        {
          output_cbor_file.setFileName(parser.value(cborOutputOption));
          if(!output_cbor_file.open(QIODevice::WriteOnly))
            {
              throw pappso::PappsoException(
                QObject::tr("Unable to open PSM CBOR output file '%1' :\n%2\n")
                  .arg(QFileInfo(output_cbor_file).absoluteFilePath())
                  .arg(output_cbor_file.errorString()));
            }
        }


      QString sageJsonFileStr = parser.value(sageJsonInputOption);

      if(!sageJsonFileStr.isEmpty())
        {
          // build a spectrum collection from mz file

          qDebug();
          monitor.setStatus(QObject::tr("getting Sage JSON result file %1")
                              .arg(sageJsonFileStr));

          QFileInfo sage_json_file(sageJsonFileStr);

          pappso::cbor::CborStreamWriter writer(&output_cbor_file);


          pappso::cbor::psm::SageFileReader sage_file_reader(
            &monitor, &writer, sage_json_file);

          qDebug();
        }

      qDebug();
      // xic_writer.write(xic);
      output_cbor_file.close();


      qDebug();
    }
  catch(pappso::PappsoException &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.qwhat());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }

  catch(std::exception &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.what());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }


  // you must call quit when complete or the program will stay in the
  // messaging loop
  quit();
}


int
main(int argc, char **argv)
{
  umask(0);
  // QTextStream consoleErr(stderr);
  // QTextStream consoleOut(stdout, QIODevice::WriteOnly);
  // ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
  // ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
  qDebug();
  QCoreApplication app(argc, argv);
  qDebug();
  QCoreApplication::setApplicationName("mzid2cbor");
  QCoreApplication::setApplicationVersion(GP_VERSION);
  QLocale::setDefault(QLocale::system());

  // create the main class
  SageToCborCli myMain;
  // connect up the signals
  QObject::connect(&myMain, SIGNAL(finished()), &app, SLOT(quit()));
  QObject::connect(
    &app, SIGNAL(aboutToQuit()), &myMain, SLOT(aboutToQuitApp()));
  qDebug();


  // This code will start the messaging engine in QT and in
  // 10ms it will start the execution in the MainClass.run routine;
  QTimer::singleShot(10, &myMain, SLOT(run()));
  return app.exec();
}
