/**
 * \file peptidoms.cpp
 * \date 12/09/2025
 * \author Olivier Langella
 * \brief CLI tool for PeptidOMS project
 */


/*
 * PeptidOMS, Spectra to protein alignment tool
 * Copyright (C) 2025  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#include "mzidtocborcli.h"
#include <sys/stat.h>
#include <QIODevice>
#include <QFile>
#include <QFileInfo>
#include <QCommandLineParser>
#include <pappsomspp/core/pappsoexception.h>
#include <pappsomspp/core/processing/uimonitor/uimonitortext.h>
#include <pappsomspp/core/processing/cbor/psm/mzidentml/mzidentmlreader.h>
#include "../core/gp_lib_config.h"


MzidToCborCli::MzidToCborCli(QObject *parent) : QObject(parent)
{
  // get the instance of the main application
  app = QCoreApplication::instance();
  // setup everything here
  // create any global objects
  // setup debug and warning mode
}


MzidToCborCli::~MzidToCborCli()
{
}

// call this routine to quit the application
void
MzidToCborCli::quit()
{
  // you can do some cleanup here
  // then do emit finished to signal CoreApplication to quit
  emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void
MzidToCborCli::aboutToQuitApp()
{
  // stop threads
  // sleep(1);   // wait for threads to stop.
  // delete any objects
}


// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void
MzidToCborCli::run()
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  //./src/peptidoms
  // -m
  // /gorgone/pappso/data_extraction_pappso/mzML/20120906_balliau_extract_1_A01_urnb-1.mzML
  // -f /gorgone/pappso/data_extraction_pappso/database/Genome_Z_mays_5a.fasta
  // -p params.json
  // -c 10


  QTextStream errorStream(stderr, QIODevice::WriteOnly);
  QTextStream outputStream(stderr, QIODevice::WriteOnly);

  try
    {
      qDebug();
      QCommandLineParser parser;
      pappso::UiMonitorText monitor(errorStream);

      // throw pappso::PappsoException("test");
      parser.setApplicationDescription(
        QString("%1 ")
          .arg(QCoreApplication::applicationName())
          .append("mzid2cbor")
          .append(" (")
          .append(GP_VERSION)
          .append(")")
          .append(" -- convert mzid file to CBOR PSM"));

      parser.addHelpOption();
      parser.addVersionOption();

      QCommandLineOption mzidInputOption(
        QStringList() << "i"
                      << "input",
        QCoreApplication::translate("MzIdentML file",
                                    "path to the MzIdentML input file <mzid>."),
        QCoreApplication::translate("MzIdentML", "MzIdentML"));

      QCommandLineOption cborOutputOption(
        QStringList() << "o",
        QCoreApplication::translate(
          "PSM cbor", "path to the PSM cbor output file <output>."),
        QCoreApplication::translate("output", "output"));


      parser.addOption(mzidInputOption);
      parser.addOption(cborOutputOption);
      qDebug();

      // Process the actual command line arguments given by the user
      parser.process(*app);

      // QCoreApplication * app(this);
      // Add your main code here
      // qDebug();
      const QStringList args = parser.positionalArguments();


      qDebug();


      qDebug();
      QFile output_cbor_file;
      if(!parser.isSet(cborOutputOption))
        {
          output_cbor_file.open(stdout, QIODevice::WriteOnly);
        }
      else
        {
          output_cbor_file.setFileName(parser.value(cborOutputOption));
          if(!output_cbor_file.open(QIODevice::WriteOnly))
            {
              throw pappso::PappsoException(
                QObject::tr("Unable to open PSM CBOR output file '%1' :\n%2\n")
                  .arg(QFileInfo(output_cbor_file).absoluteFilePath())
                  .arg(output_cbor_file.errorString()));
            }
        }
        
    


      QString mzidFileStr = parser.value(mzidInputOption);

      if(!mzidFileStr.isEmpty())
        {
          // build a spectrum collection from mz file

          qDebug();
          monitor.setStatus(
            QObject::tr("getting MzIdentML file %1").arg(mzidFileStr));

    QFileInfo mzid_file(mzidFileStr);

    pappso::cbor::CborStreamWriter writer(&output_cbor_file);


    pappso::cbor::psm::MzIdentMlReader mzid_file_reader(&monitor, &writer, mzid_file);

          qDebug();
        }

      qDebug();
      // xic_writer.write(xic);
      output_cbor_file.close();


      qDebug();
    }
  catch(pappso::PappsoException &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.qwhat());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }

  catch(std::exception &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.what());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }


  // you must call quit when complete or the program will stay in the
  // messaging loop
  quit();
}


int
main(int argc, char **argv)
{
  umask(0);
  // QTextStream consoleErr(stderr);
  // QTextStream consoleOut(stdout, QIODevice::WriteOnly);
  // ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
  // ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
  qDebug();
  QCoreApplication app(argc, argv);
  qDebug();
  QCoreApplication::setApplicationName("mzid2cbor");
  QCoreApplication::setApplicationVersion(GP_VERSION);
  QLocale::setDefault(QLocale::system());

  // create the main class
  MzidToCborCli myMain;
  // connect up the signals
  QObject::connect(&myMain, SIGNAL(finished()), &app, SLOT(quit()));
  QObject::connect(
    &app, SIGNAL(aboutToQuit()), &myMain, SLOT(aboutToQuitApp()));
  qDebug();


  // This code will start the messaging engine in QT and in
  // 10ms it will start the execution in the MainClass.run routine;
  QTimer::singleShot(10, &myMain, SLOT(run()));
  return app.exec();
}
