import logging
import os
import re
from json import dumps, loads

from pstatic.diannParam import PARAM

logging.basicConfig()
logging.root.setLevel(logging.NOTSET)
logging.basicConfig(level=logging.NOTSET)

logger = logging.getLogger(__name__)
FORMAT = "[%(filename)s:%(lineno)s - %(funcName)20s() ] %(message)s"
logging.basicConfig(format=FORMAT)
logger.setLevel(logging.INFO)


class configDiann:
    # init function instantiate self.param and
    # self.paramWith Arg to store param
    def __init__(self):
        self._param = []
        self._paramWithArg = {}
        self._rawFiles = []
        self._fastaFiles = []
        self._originalFile = ""
        self._rawFileDirectory = ""
        self._outputDirectory = ""

    # function to read diann config file
    # configFile path and readability must be check before and in full path format
    def readCfg(self, CfgFile):
        self._originalFile = CfgFile
        configFile = open(CfgFile, "r")
        unImportLine = []
        for line in configFile:
            tabline = line.strip().split(" ")
            tabline[0] = re.sub(r"^--", "", tabline[0])
            if self._checkParam(tabline[0]):
                if len(tabline) == 1:
                    self._param.append(re.sub(r"^--", "", tabline[0]))
                else:
                    tabline[1] = re.sub(r'^"|"$', "", tabline[1])
                    if tabline[0] == "f":
                        self.addSample(tabline[1])
                    elif tabline[0] == "fasta":
                        self.addFastaFile(tabline[1])
                    if tabline[0] in self._paramWithArg.keys():
                        self._paramWithArg[re.sub(r"^--", "", tabline[0])].append(
                            tabline[1]
                        )
                    else:
                        self._paramWithArg[re.sub(r"^--", "", tabline[0])] = [
                            tabline[1]
                        ]
            else:
                unImportLine.append(line)
                logging.debug(f"param '{line.strip()}'" + "is not a valid parameter")

    # function to add samples before adding test if sample is already set
    # sample must be specified in fullpath
    # availability of the file must be check before

    def addSamples(self, sampleList):
        for sample in sampleList:
            if sample not in self._rawFiles:
                self._rawFiles.append(sample)
            else:
                log = f"sample {sample} is already set. it was ignored"
                logging.debug(log)

    def addSample(self, sample):
        if sample not in self._rawFiles:
            self._rawFiles.append(sample)
        else:
            logging.debug(f"sample {sample} is already set. it was ignored")

    # function to add fastafile before adding test if fasta is already set
    # fastafile must be specified in fullpath
    # availability of the file must be check before

    def addFastaFiles(self, fastaFileList):
        for fastafile in fastaFileList:
            if fastafile not in self._fastaFiles:
                self._fastaFiles.append(fastafile)
            else:
                log = f"sample {fastafile} is already set. it was ignored"
                logging.debug(log)

    def addFastaFile(self, fastafile):
        if fastafile not in self._fastaFiles:
            self._fastaFiles.append(fastafile)
        else:
            logging.debug(f"sample {fastafile} is already set. it was ignored")

    def setRawFileDirectory(self, directory):
        self._rawFileDirectory = directory

    def setoutputDirectory(self, directory):
        self._outputDirectory = directory
        logging.debug(f"{type(directory)}")

    # getters

    def getOriginalFile(self):
        return self._originalFile

    def getParamWithArg(self, param):
        return self._paramWithArg[param]

    def getRawFileList(self):
        return self._rawFiles

    def getFastaFileList(self):
        return self._fastaFiles

    def getRawFileDirectory(self):
        return self._rawFileDirectory

    def getOutputDirectory(self):
        return self._outputDirectory

    def hasParam(self, param):
        if self._checkParam(param):
            if PARAM[param]["arg"] == "":
                logging.debug("param is without arg")
                if param in self._param:
                    logging.debug(f"{param} is present in config")
                    return True
                else:
                    logging.debug(f"{param} is not present in config")
                    return False
            else:
                logging.debug("param is with arg")
                if param in self._paramWithArg:
                    logging.debug(f"{param} is present in config")
                    return True
                else:
                    logging.debug(f"{param} is not present in config")
                    return False

    # internal function to test if a parameter is available or not
    # @param the param to check
    # @return bool
    def _checkParam(self, param):
        if param in PARAM.keys():
            return True
        elif re.search(r"unimod[0-9]+", param) is not None:
            return True
        else:
            return False

    # function to add param
    # @param param : a valid parameter for diann config file
    def addParam(self, param):
        if self._checkParam(param):
            if param not in self._param:
                self._param.append(param)
            else:
                logging.debug(f"{param} is already present in parameter")
        else:
            logging.debug(f"param '{param}' is not a valid parameter")

    # function to add param with aprg
    # @param param : a valid parameter for diann config file
    # @param arg : the argument to associate to parameter
    def addParamWithArg(self, param, arg):
        ## workaround for var mod probably to implement also for fixed-mod
        if param == "var-mod":
            self._paramWithArg[param].append(arg)
        elif self._checkParam(param):
            if param not in self._paramWithArg.keys():
                self._paramWithArg[param] = [arg]
            elif arg in self._paramWithArg[param]:
                logging.debug(
                    f"{param} is already present" + f" in parameter with argument {arg}"
                )
            else:
                self._paramWithArg[param].append(arg)
                logging.debug(f"{param} is added to {param}")
        else:
            logging.debug(f"param '{param}' is not a valid parameter")

    # function to modify a argument for a given argument
    # @param param : a valid parameter for diann config file
    # @param oldarg : the argument to change
    # @param newarg : the new value for the argument
    def modifyArgForParam(self, param, oldarg, newarg):
        ## workaround for var mod probably to implement also for fixed-mod
        if param == "var-mod":
            tabmod = newarg.split("; ")
            self._paramWithArg[param] = tabmod
        elif self._checkParam(param):
            if param not in self._paramWithArg.keys():
                logging.debug(f"{param} is not define in paramWithArg")
            elif oldarg in self._paramWithArg[param]:
                self._paramWithArg[param].remove(oldarg)
                self._paramWithArg[param].append(newarg)
                logging.debug(f"{param} arg is modify " + f"from {oldarg} to {newarg}")
            else:
                logging.debug(f"{param} arg does not have {oldarg}")
        else:
            logging.debug(f"param '{param}' is not a valid parameter")

    # function to remove a given argument for a parameter
    # @param param the name of the parameter where arg need to be removed
    # @param arg the arg to remove
    def removeArgForParam(self, param, arg):
        if param not in self._paramWithArg.keys():
            logging.debug(f"{param} is not define in paramWithArg")
        elif arg in self._paramWithArg[param]:
            self._paramWithArg[param].remove(arg)
            logging.debug(f"{arg} is removed from {param}")
            if len(self._paramWithArg[param]) == 0:
                del self._paramWithArg[param]
                logging.debug(f"{param} is removed from paramWithArg")

    # function to remove param from param or paramwitharg
    # @param param to remove
    def removeParam(self, param):
        if param in self._param:
            self._param.remove(param)
        elif param in self._paramWithArg.keys():
            self._paramWithArg.pop(param)
        else:
            logging.info(param + "is not set in configDiann object")

    # function to test if param has arg
    # @param param the name of the parameter where arg need to be removed
    # @param arg the arg to remove
    def isParamWithArg(self, param):
        # if param in self._paramWithArg.keys():
        if PARAM[param]["arg"] != "":
            return True
        else:
            return False

    # generate json string to write to file
    def toJSON(self):
        export = dict()
        export["param"] = self._param
        export["paramWithArg"] = self._paramWithArg
        export["rawFiles"] = self._rawFiles
        export["fastaFiles"] = self._fastaFiles
        export["originalFile"] = self._originalFile
        export["rawFileDirectory"] = self._rawFileDirectory
        export["outputDirectory"] = self._outputDirectory
        logging.info(export)
        return dumps(export, indent=4, sort_keys=True)

    # read json string to load data
    # @param jsonstring : a string read from a json files
    def fromJSON(self, jsonstring):
        imported = loads(jsonstring)
        logging.debug(imported)
        if "param" in imported.keys():
            self._param = imported["param"]
            log = f"{len(self._param)} parameter imported from param key"
            logging.debug(log)
        if "paramWithArg" in imported.keys():
            self._paramWithArg = imported["paramWithArg"]
            logging.debug(
                f"{len(self._paramWithArg)} parameter"
                + " imported from paramWithArg key"
            )
        if "rawFiles" in imported.keys():
            self._rawFiles = imported["rawFiles"]
            logging.debug(
                f"{len(self._rawFiles)} rawfiles" + " imported from rawFiles key"
            )
        if "fastaFiles" in imported.keys():
            self._fastaFiles = imported["fastaFiles"]
            logging.debug(
                f"{len(self._fastaFiles)} fastafiles" + " imported from fastafiles key"
            )
        if "originalFile" in imported.keys():
            self._originalFile = imported["originalFile"]
            logging.debug("originalFile name imported from originalFile key")

    # generate dianncfg string
    def toCfg(self):
        export = ""
        if len(self._param) > 0:
            for param in self._param:
                export = export + f"--{param}\n"
        if len(self._paramWithArg.keys()) > 1:
            for paramkey in self._paramWithArg.keys():
                for arg in self._paramWithArg[paramkey]:
                    if paramkey in ["out", "out-lib"]:
                        export = export + f"--{paramkey}"
                        export = (
                            export
                            + f" {os.path.join(self._outputDirectory, paramkey)}\n"
                        )
                    else:
                        export = export + f"--{paramkey}"
                        export = export + f" {arg}\n"
        if len(self._fastaFiles) > 0:
            for fastafiles in self._fastaFiles:
                export = export + f"--fasta {fastafiles}\n"
        if len(self._rawFiles) > 0:
            for rawfile in self._rawFiles:
                export = (
                    export + f"--f {os.path.join(self._rawFileDirectory, rawfile)}\n"
                )
        return export

    # return a dict with all param (without file and fasta)

    def toDict(self):
        export = {}
        for param in self._param:
            export[param] = [""]
        for key in self._paramWithArg.keys():
            export[key] = self._paramWithArg[key]
        return export

    def setCfgFilePath(self, pathFile):
        self._originalFile = pathFile
