# This Python file uses the following encoding: utf-8
import logging
import os
import sys

from PyQt5.QtWidgets import QApplication, QFileDialog, QMainWindow

# Important:
# You need to run the following command to generate the ui_form.py file
#     pyside6-uic form.ui -o ui_form.py, or
#     pyside2-uic form.ui -o ui_form.py
from diacmdbuilder.ui_main import Ui_MainWindow
from pstatic.slurmCfg import getCommandLine
from pstruct.configDiann import configDiann
from pwidget.paramEditor import paramEditor
from pwidget.pfileDialog import pfileDialog
from QT.settings import addSetting, getSetting, init

logging.basicConfig()
logging.root.setLevel(logging.NOTSET)
logging.basicConfig(level=logging.NOTSET)

logger = logging.getLogger(__name__)
FORMAT = "[%(filename)s:%(lineno)s - %(funcName)20s() ] %(message)s"
logging.basicConfig(format=FORMAT)
logger.setLevel(logging.INFO)


class MainWindow(QMainWindow):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.ui = Ui_MainWindow()
        self.ui.setupUi(self)
        self.cfgFileLoaded = False
        self.rawFileLoaded = False
        self.rawFileLocationsetted = False
        self.outputDirectorySetted = False
        self.fastaFileList = []
        self.fastaFileLoaded = True
        self.myconfig = configDiann()
        init()
        self.ui.SlurmCommandLine.setText(getCommandLine())
        self.ui.rawFileLocationDir.setText(getSetting("param", "rawdirectory"))

    # load cfg file and parse it using configDiann class
    def loadCfg(self):
        logging.info("cfgLoader clicked")
        cfgfile = QFileDialog.getOpenFileName(
            self,
            "Select a cfg file",
            getSetting("param", "cfgdir"),
            "DiannConfig file (*.cfg);" + ";all files (*.*)",
        )[0]
        if os.path.isfile(cfgfile):
            logging.info(f"CfgFile : {cfgfile}")
            # self.myconfig = configDiann()
            self.myconfig.readCfg(cfgfile)
            logging.debug(f"loaded info : {self.myconfig.toJSON()}")
            self.ui.cfgLoaded.setText(cfgfile)
            self.cfgFileLoaded = True
            dirCfg = os.path.dirname(cfgfile)
            addSetting("param", "cfgdir", dirCfg)
        else:
            logging.info(f"{cfgfile} does not exist.")

    def loadRawFile(self):
        logging.info("rawFileListLoader clicked")
        currentDirectory = getSetting("param", "rawdirectory")
        self.ui_ftp = pfileDialog(currentDirectory, True, False)
        self.ui_ftp.selectDirectoryEvent.connect(self.setRawList)
        self.ui_ftp.show()

    def selectOutputDirectory(self):
        logging.info("outputDirectoryLoader clicked")
        OutputDirectory = QFileDialog.getExistingDirectory(
            self, "Select output " + "directory", getSetting("param", "outputdir")
        )
        logging.info(f"{OutputDirectory}")
        if os.path.exists(OutputDirectory):
            self.ui.outputDirLocation.setText(OutputDirectory)
            self.myconfig.setoutputDirectory(OutputDirectory)
            self.outputDirectorySetted = True
            addSetting("param", "outputdir", OutputDirectory)
        else:
            logging.info(f"{OutputDirectory} directory does not exists")

    def selectFastaFile(self):
        logging.info("FastaFileLoader clicked")
        fastaFiles = QFileDialog.getOpenFileNames(
            self,
            "Select fasta file listing",
            getSetting("param", "fastadir"),
            "all files (*.*)",
        )[0]
        logging.info(fastaFiles)
        for fastaFile in fastaFiles:
            if fastaFile not in self.fastaFileList:
                if os.path.isfile(fastaFile):
                    self.fastaFileList.append(fastaFile)
                    self.myconfig.addFastaFile(fastaFile)
                    self.fastaFileLoaded = True
                    fastaString = "\n* ".join(self.fastaFileList)
                    self.ui.FastaFileList.setText(f"* {fastaString}")
                    fastaDir = os.path.dirname(fastaFile)
                    logging.info(f"{fastaFile} is added")
                    addSetting("param", "fastadir", fastaDir)
                else:
                    logging.info(f"{fastaFile} does not exists")
            else:
                logging.info(f"{fastaFile} is alreadySet")

    def generateCMD(self):
        if self.ui.libGenSwitch.isChecked():
            if all(
                [self.cfgFileLoaded, self.outputDirectorySetted, self.fastaFileLoaded]
            ):
                outputCfgfilename = os.path.join(
                    self.myconfig.getOutputDirectory(), "outputCfg.cfg"
                )
                outputfile = open(outputCfgfilename, "w")
                outputfile.write(self.myconfig.toCfg())
                outputfile.close()
                self.ui.cfgFileName.setText(outputCfgfilename)
                cmd = self.ui.SlurmCommandLine.text()
                # cmd += getSetting("param", "diannBin")
                cmd += " /opt/diann/diann-2.0.2/diann-linux "
                cmd += f"--cfg {outputCfgfilename}"
                commandFileName = os.path.join(
                    self.myconfig.getOutputDirectory(), "commandLine.txt"
                )
                commandFile = open(commandFileName, "w")
                commandFile.write(cmd)
                commandFile.close()
                self.ui.cmdGorgone.setText(commandFileName)
            else:
                logger.info("some information needed not define")
        else:
            if all(
                [
                    self.cfgFileLoaded,
                    self.rawFileLocationsetted,
                    self.outputDirectorySetted,
                    self.fastaFileLoaded,
                ]
            ):
                outputCfgfilename = os.path.join(
                    self.myconfig.getOutputDirectory(), "outputCfg.cfg"
                )
                outputfile = open(outputCfgfilename, "w")
                outputfile.write(self.myconfig.toCfg())
                outputfile.close()
                self.ui.cfgFileName.setText(outputCfgfilename)
                cmd = self.ui.SlurmCommandLine.text()
                cmd += " /opt/diann/diann-2.0.2/diann-linux "
                cmd += f"--cfg {outputCfgfilename}"
                commandFileName = os.path.join(
                    self.myconfig.getOutputDirectory(), "commandLine.txt"
                )
                commandFile = open(commandFileName, "w")
                commandFile.write(cmd)
                commandFile.close()
                self.ui.cmdGorgone.setText(commandFileName)

    def generateJson(self):
        if self.outputDirectorySetted:
            JsonString = self.myconfig.toJSON()
            fileName = os.path.join(
                self.myconfig.getOutputDirectory(), "outputcfg.json"
            )
            JsonFile = open(fileName, "w")
            JsonFile.write(JsonString)
            JsonFile.close()
            self.ui.JsonFileName.setText(fileName)

    def setRawList(self, directories):
        rawList = []
        for directory in directories:
            rawList.append(directory[0])
        self.rawFileLoaded = True
        rawListString = ""
        for item in rawList:
            rawListString = rawListString + item + "\n* "
        self.ui.rawfileListName.setText(f"* {rawListString}")
        logging.info(rawList)
        # directory = rawListString[0][:rawListString[0].rindex("/")]
        # self.ui.rawFileLocationDir.setText(directory)
        self.myconfig.addSamples(rawList)
        self.ui.rawFileLocationDir.setText(self.ui_ftp.getCurrentDirectory())
        self.myconfig.setRawFileDirectory(self.ui_ftp.getCurrentDirectory())
        addSetting("param", "rawdirectory", self.ui_ftp.getCurrentDirectory())
        self.rawFileLocationsetted = True

    # Show config and edit them
    def showCfg(self):
        try:
            self.ui_cfgEditor = paramEditor(self.myconfig)
        except NameError:
            self.ui_cfgEditor = paramEditor(None)
        else:
            self.ui_cfgEditor.selectAcceptEvent.connect(self.setCfg)
            self.ui_cfgEditor.show()

    def setCfg(self, cfg):
        if cfg != -1:
            self.myconfig = cfg
            logging.info(self.myconfig.toJSON())
            self.ui.cfgLoaded.setText(self.myconfig.getOriginalFile())
            self.cfgFileLoaded = True


if __name__ == "__main__":
    app = QApplication(sys.argv)
    widget = MainWindow()
    widget.show()
    sys.exit(app.exec())
